# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import os
import socket
import uuid
from threading import Condition

from mongolock import MongoLock

from common.db.mongo import databases
from travel.rasp.library.python.common23.utils.code_utils import ContextManagerAsDecorator
from common.utils.threadutils import Ticker

LOCK_COLLECTION_NAME = 'locks'


class Lock(ContextManagerAsDecorator):
    """
    Lock a task by name in mongo.

    @Lock('lock_name')
    def foo():
        pass

    with Lock('lock_name'):
        pass

    """

    def __init__(self, name, collection=LOCK_COLLECTION_NAME, timeout=None, expire=10, owner=None, renew_interval=5,
                 updater_timeout=5, notify_about_renewals=False, database_name='default'):
        self.name = name
        self.collection = collection
        self.timeout = timeout
        self.database_name = database_name

        if expire and renew_interval:
            assert renew_interval < expire

        self.expire = expire
        self.renew_interval = renew_interval
        self.updater_timeout = updater_timeout

        if not owner:
            owner = '{}__{}__{}'.format(socket.getfqdn(), os.getpid(), str(uuid.uuid4()))
        self.owner = owner

        self.lock = MongoLock(collection=getattr(databases[database_name], self.collection))
        self.lock_context = None
        self.lock_updater = None
        if notify_about_renewals:
            self.lock_update_condition = Condition()
        else:
            self.lock_update_condition = None

    def __enter__(self):
        self.lock_context = self.lock(self.name, self.owner, timeout=self.timeout, expire=self.expire)
        self.lock_context.__enter__()

        if self.expire and self.renew_interval:
            self.lock_updater = Ticker(self.renew_interval, target=self.update_lock)
            self.lock_updater.setDaemon(True)
            self.lock_updater.start()

        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.lock_updater:
            self.lock_updater.stop()
            self.lock_updater.join(timeout=self.updater_timeout)
            if self.lock_updater.is_alive():
                raise Exception("Unable to stop lock updater thread, it's stuck somewhere inside.")

        self.lock_context.__exit__(exc_type, exc_val, exc_tb)

    def _renew_lock(self):
        self.lock.touch(self.name, self.owner, self.expire)

    def update_lock(self):
        if self.lock_update_condition:
            with self.lock_update_condition:
                self._renew_lock()
                self.lock_update_condition.notify_all()
        else:
            self._renew_lock()


lock = Lock
