# coding: utf8
from __future__ import print_function

import calendar
import json
import os
import threading
from datetime import datetime
from stat import ST_DEV, ST_INO

from django.utils.encoding import smart_bytes, force_bytes


def is_v6(ip):
    groups = ip.split(':')

    for group in groups:
        if group == '':
            continue

        try:
            int(group, 16)
        except ValueError:
            return False

    return True


class Log(object):
    u"""Класс для логгирования переходов в формате маркетстата"""

    def __init__(self, log_file_name):
        self.stream = None
        self.mode = 'a'

        self.baseFilename = log_file_name
        self.dirname = os.path.dirname(log_file_name)

        if not os.path.exists(self.baseFilename):
            self.dev, self.ino = -1, -1
        else:
            stat = os.stat(self.baseFilename)
            self.dev, self.ino = stat[ST_DEV], stat[ST_INO]

        self.write_lock = threading.Lock()

    def open(self):
        if not os.path.exists(self.dirname):
            os.makedirs(self.dirname)

        return open(self.baseFilename, self.mode)

    def _write(self, message):
        if not os.path.exists(self.baseFilename):
            stat = None
            changed = 1

        else:
            stat = os.stat(self.baseFilename)
            changed = (stat[ST_DEV] != self.dev) or (stat[ST_INO] != self.ino)

        if changed and self.stream is not None:
            if not self.stream.closed:
                self.stream.flush()
                self.stream.close()

            self.stream = open(self.baseFilename, self.mode)

            if stat is None:
                stat = os.stat(self.baseFilename)

            self.dev, self.ino = stat[ST_DEV], stat[ST_INO]

        if self.stream is None:
            self.stream = self.open()

        print(message, file=self.stream)

        self.stream.flush()

    def write(self, message):
        with self.write_lock:
            self._write(message)

    def log_line(self, data):
        return '@@'.join('%s=%s' % (k, v or '') for k, v in data.items())

    def user_ip(self, request):
        return request.META.get('REMOTE_ADDR', None)

    def process_userip(self, userip):
        if userip:
            if userip.startswith('::ffff:'):
                userip = userip[7:]

            elif is_v6(userip):
                # RASP-10956 - Писать в редир.лог фэйковый IPv4 адрес вместо IPv6
                userip = '8.8.8.8'

        return userip

    def log(self, request, extra):
        self.write_event(
            yandexuid=request.COOKIES.get('yandexuid', None),
            passportuid=getattr(request, 'yauser', None) and request.yauser.uid,
            userip=self.user_ip(request),
            extra=extra
        )

    def write_event(self, yandexuid=None, passportuid=None, userip=None, extra=None):
        data = {
            'eventtime': datetime.utcnow().strftime('%Y%m%d%H%M%S'),
            'yandexuid': yandexuid,
            'passportuid': passportuid,
            'userip': self.process_userip(userip),
        }

        if extra:
            data.update(extra)

        message = self.log_line(data)

        self.write(smart_bytes(message))


class DsvLog(Log):
    def process_userip(self, userip):
        if userip and userip.startswith('::ffff:'):
            userip = userip[7:]

        return userip

    def log_line(self, data):
        tskv_header = 'tskv\ttskv_format=%s\tunixtime=%s' % (
            data.pop('tskv_format', None),
            calendar.timegm(datetime.utcnow().timetuple())
        )

        return '%s\t%s' % (
            tskv_header,
            "\t".join('%s=%s' % (k, v or '') for k, v in data.items())
        )

    def user_ip(self, request):
        return request.META.get('REMOTE_ADDR', None)


class DsvSimpleLog(DsvLog):
    u"""DSV-лог без дополнений данными"""

    def log(self, data):
        message = self.log_line(data)

        self.write(message)


class JsonLog(Log):
    def log_line(self, data):

        return force_bytes(json.dumps(data, ensure_ascii=False))

    def log(self, data):
        message = self.log_line(data)

        self.write(message)


class AuctionLog(Log):

    def log(self, request, extra):

        data = {
            'time': datetime.utcnow().strftime('%Y%m%d%H%M%S')
        }

        data.update(extra)

        message = self.log_line(data)
        self.write(message)
