# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import time as os_time
from datetime import timedelta
from math import floor


class PercentageStatus(object):
    u"""
    Класс который позволяет считать в процентах, оставшееся время.
    Счетчик приспособлен для целых чисел, хотя возможно будет работать и с float
    """
    def __init__(self, total, log=None, show_elapsed=True, show_remained=True,
                 callback=None, print_to_stream=None, title=u''):
        u"""
        @param total: всего элементов или событий,
        @param log: лог куда будет писаться процентовка,
        @param show_elapsed: записывать в лог сколько прошло времени,
        @param show_remained: записывать в лог сколько примерно осталось времени,
        @param callback: функция, которая будет вызываться при смене процента,
            в качестве аргумента функция должна принимать объект типа PercentageStatus.
            Так же функции можно передать дополнительные аргументы в step.
        """
        self.total = total if total > 0 else False
        self.log = log
        self.show_elapsed = show_elapsed
        self.show_remained = show_remained
        self.callback = callback
        self.start_time = os_time.time()
        self.previous_pos = 0
        self.previous_percentage = 0
        self.pos = 0
        self.percentage = 0
        self.print_to_stream = print_to_stream
        self.title = title

    def restart(self):
        u"""
        Сбросить статус, и начать отсчет заново.
        """
        self.start_time = os_time.time()
        self.pos = 0
        self.percentage = 0
        self.previous_pos = 0
        self.previous_percentage = 0

    def status_changed(self):
        return floor(self.percentage) != floor(self.previous_percentage) or \
            self.pos == self.total

    def step(self, increment=1, *args, **kwargs):
        u"""
        @param increment: увеличить счетчик на эту величину, по умолчанию 1.
        Так же можно передать параметры для callback.
        """
        if not self.total:
            return
        self.pos = self.previous_pos + increment
        self.percentage = 100.0 * self.pos / self.total
        # Сменилась процентовка, пишем в лог
        if self.status_changed():
            self.time_elapsed = os_time.time() - self.start_time
            self.time_per_item = self.time_elapsed / self.pos
            self.time_remained = self.time_per_item * self.total - self.time_elapsed
            self.timedelta_elapsed = timedelta(seconds=int(self.time_elapsed))
            self.timedelta_remained = timedelta(seconds=int(self.time_remained))

            if self.log or self.print_to_stream:
                format = []
                if self.title:
                    format = [self.title]

                format += [u"Progress:%(percentage)4.0f%%"]
                if self.show_elapsed:
                    format += [u"Elapsed: %(timedelta_elapsed)s"]
                if self.show_remained:
                    format += [u"Remained: %(timedelta_remained)s"]
                format = u". ".join(format)
                if self.log:
                    self.log.info(format, vars(self))
                if self.print_to_stream:
                    print(format % vars(self), file=self.print_to_stream)

            if callable(self.callback):
                self.callback(self, *args, **kwargs)

        self.previous_pos = self.pos
        self.previous_percentage = self.percentage
