# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import time as c_time
from functools import wraps

from django.conf import settings

from common.settings.utils import define_setting

MAX_RPS = 10
define_setting('REQUEST_LIMITER_NEVER_SLEEP', default=False)


class RequestLimiter(object):
    """
    Гарантирует RPS не выше заданного (при правильном использовании, конечно)
    Перед каждым запросом нужно вызывать make_time_delay.

    Примеры использования:

    request_limiter = RequestLimiter(max_rps=10)
    for url in urls:
        request_limiter.make_time_delay()
        data = requests.get(url)

    или

    request_limiter = RequestLimiter(max_rps=10)

    @request_limiter.throttle
    def get_content(url)
        return requests.get(url)
    """
    def __init__(self, max_rps):
        self.delay = 1. / max_rps
        self._last_request_time = None

    def make_time_delay(self):
        now = c_time.time()
        if self._last_request_time is not None:
            elapsed = now - self._last_request_time
            if elapsed < self.delay:
                if not settings.REQUEST_LIMITER_NEVER_SLEEP:
                    c_time.sleep(self.delay - elapsed)

        self._last_request_time = now

    def throttle(self, func):
        """
        Декоратор для ограничения количества вызовов функции в секунду.
        Для правильного ограничения количества запросов в секунду, в декорируемой функции должен быть один запрос.
        """
        @wraps(func)
        def wrapper(*args, **kwargs):
            self.make_time_delay()
            return func(*args, **kwargs)
        return wrapper
