# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import itertools
from collections import defaultdict

from django.db.models import Q
from django.utils.http import urlencode
from six.moves.urllib_parse import quote_plus

from common.models.geo import Station, Station2Settlement, StationMajority
from common.models.transport import TransportType


YMAP_BASE_URLS = {
    'ru': 'https://maps.yandex.ru',
    'ua': 'https://maps.yandex.ua',
    'by': 'https://maps.yandex.by',
    'kz': 'https://maps.yandex.kz',
    'uz': 'https://maps.yandex.uz',
}


def get_ymap_url(settlement, national_version=None, lang=None):
    """
    Получает URL для сервиса Yandex.Maps, по которому будет открыта карта с данным населенным пунктом.
    1. Если у населенный пункта есть ссылка на геобазу, то результат будет в формате /213/moscow/
    2. Если ссылки на геобазу нет, то результат будет в формате /?text=Волоколамск, Волоколамский район,
        Московская область, Россия
    Во втором случае значение GET-параметра text вычисляется с учетом национальной версии
    (нужна для спорных территорий) и языка.
    """

    base_url_ru = YMAP_BASE_URLS['ru']
    base_url = YMAP_BASE_URLS.get(national_version, base_url_ru) if national_version else base_url_ru

    geobase_region = settlement.get_geobase_region()
    if geobase_region and geobase_region.ename:
        geobase_name = quote_plus(geobase_region.ename.replace(b' ', b'-').lower())
        return '{}/{}/{}/'.format(base_url, geobase_region.id, geobase_name)

    query = {
        'text': settlement.L_title_with_full_geography(national_version, lang)
    }
    return '{}/?{}'.format(base_url, urlencode(query))


def fetch_station_settlement_ids():
    station_settlement_ids = defaultdict(set)
    for station_id, settlement_id in itertools.chain(
        Station.objects.exclude(settlement_id=None).values_list('id', 'settlement_id'),
        Station2Settlement.objects.values_list('station_id', 'settlement_id')
    ):
        station_settlement_ids[station_id].add(settlement_id)
    return dict(station_settlement_ids)


def fetch_station_settlement_ids_with_none():
    station_settlement_ids = defaultdict(set)
    for station_id, settlement_id in itertools.chain(
        Station.objects.values_list('id', 'settlement_id'),
        Station2Settlement.objects.values_list('station_id', 'settlement_id')
    ):
        station_settlement_ids[station_id].add(settlement_id)
    return dict(station_settlement_ids)


def fill_best_rts_for_settlement(rtstations, station2settlement_ids=None):
    if not station2settlement_ids:
        station2settlement_ids = fetch_station_settlement_ids()

    best_rts_for_start_from_settlement = dict()
    best_rts_for_finish_in_settlement = dict()

    for rts in rtstations:
        if rts.tz_departure == rts.tz_arrival:
            continue

        settlement_ids = station2settlement_ids[rts.station_id] if rts.station_id in station2settlement_ids else []

        if rts.is_searchable_from and rts.tz_departure is not None:
            for settlement_id in settlement_ids:
                if settlement_id not in best_rts_for_start_from_settlement:
                    best_rts_for_start_from_settlement[settlement_id] = rts
                else:
                    best_rts = best_rts_for_start_from_settlement[settlement_id]
                    # Первая станция нитки - лучшая для начала путешествия
                    if best_rts.tz_arrival is None:
                        break
                    if best_rts.station.majority_id >= rts.station.majority_id:
                        best_rts_for_start_from_settlement[settlement_id] = rts

        if rts.is_searchable_to and rts.tz_arrival is not None:
            for settlement_id in settlement_ids:
                if settlement_id not in best_rts_for_finish_in_settlement \
                        or best_rts_for_finish_in_settlement[
                            settlement_id
                        ].station.majority_id > rts.station.majority_id:
                    best_rts_for_finish_in_settlement[settlement_id] = rts

    return best_rts_for_start_from_settlement, best_rts_for_finish_in_settlement


def _iterate_settlement_stations(settlement, stations_qs, max_majority, t_type_id):
    """
    Фильтрация списка станций по типу транспорта и важности
    :param stations_qs: запрос исходного списка станций
    """
    is_foreign = settlement.is_foreign()
    if is_foreign and t_type_id and t_type_id != TransportType.PLANE_ID:
        return

    qs = Station.objects.filter(hidden=False, majority__id__lte=max_majority)

    suburban_stations = {}

    if is_foreign:
        qs = qs.filter(t_type_id=TransportType.PLANE_ID)
    elif t_type_id == TransportType.SUBURBAN_ID:
        qs = qs.filter(t_type_id=TransportType.TRAIN_ID)
    elif t_type_id:
        qs = qs.filter(t_type_id=t_type_id)
    else:
        suburban_qs = qs.filter(t_type_id=TransportType.TRAIN_ID)
        suburbans = list(suburban_qs.filter(stations_qs))
        for station in suburbans:
            type_choices = station.type_choices_set
            if 'suburban' in type_choices:
                suburban_stations[station.id] = station

    queried_stations = list(qs.filter(stations_qs))
    for station in queried_stations:
        type_choices = station.type_choices_set

        if station.t_type_id == TransportType.PLANE_ID:
            if 'tablo' in type_choices:
                yield station

        elif station.t_type_id == TransportType.TRAIN_ID:
            is_appended = False
            if (not t_type_id or t_type_id == TransportType.TRAIN_ID) and 'train' in type_choices:
                is_appended = True
                yield station
            if (
                (not t_type_id or t_type_id == TransportType.SUBURBAN_ID)
                and ('suburban' in type_choices or 'aeroexpress' in type_choices)
            ):
                # Станция может быть возвращена с типом транспорта suburban
                # Если она уже была возвращена с типом train, то она возвращается два раза
                suburban_station = suburban_stations[station.id] if is_appended else station
                suburban_station.t_type = TransportType.get_suburban_type()
                yield suburban_station
        else:
            if 'schedule' in type_choices or 'tablo' in type_choices:
                yield station


def _run_queries_list(queries, settlement, max_majority, t_type_id):
    stations_dict = defaultdict(dict)
    for query in queries:
        for station in _iterate_settlement_stations(settlement, query, max_majority, t_type_id):
            stations_dict[station.t_type_id][station.id] = station

    stations = []
    for key in stations_dict.keys():
        stations.extend(stations_dict[key].values())
    return stations


def get_connected_stations(settlement, max_majority, t_type_id=None):
    """
    Возвращает все станции города или связанные с городом
    :param settlement: город
    :param max_majority: максимальное id важности возвращаемых станций
    :param t_type_id: тип транспорта, если не указан, то возвращаются станции всех типов
    :return:
    """
    return _run_queries_list(
        [Q(settlement=settlement), Q(station2settlement__settlement=settlement)],
        settlement, max_majority, t_type_id
    )

def get_related_stations(settlement, max_majority, t_type_id=None):
    """
    Возвращает все станции, относящиеся к городу
    :param settlement: город
    :param max_majority: максимальное id важности возвращаемых станций
    :param t_type_id: тип транспорта, если не указан, то возвращаются станции всех типов
    """
    return _run_queries_list(
        [Q(related_settlements__settlement=settlement)],
        settlement, max_majority, t_type_id
    )

def get_main_stations(settlement, t_type_id=None):
    """
    Возвращает все станции как-либо связанные с городом, с важностью достаточной для попадания в табло
    :param settlement: город
    :param t_type_id: тип транспорта, если не указан, то возвращаются станции всех типов
    """
    return _run_queries_list(
        [
            Q(settlement=settlement),
            Q(station2settlement__settlement=settlement),
            Q(related_settlements__settlement=settlement)
        ],
        settlement, StationMajority.IN_TABLO_ID, t_type_id
    )
