# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import operator
import os
import signal
import subprocess
from time import sleep as time_sleep, time as time_time


log = logging.getLogger(__name__)


def signal_to_name(sig_number):
    for name, value in sorted(signal.__dict__.items(), key=operator.itemgetter(0)):
        if name.startswith('SIG') and value == sig_number:
            return name
    return None


class ProcessResult(object):
    FINISHED = 'finished'
    KILLED = 'killed'
    UNKNOWN = 'unknown'

    def __init__(self, result_type, status_code=None, reason=None, success=False):
        self.result_type = result_type
        self.status_code = status_code
        self.reason = reason
        self.success = success

    def is_success(self):
        return self.success

    def __str__(self):
        return ('<ProcessResult: {self.result_type} status_code={self.status_code} reason={self.reason}>'
                .format(self=self))


def get_result(wait_status, extra_reason=None):
    if os.WIFSIGNALED(wait_status):
        if extra_reason:
            reason = '{}. Killed with {{}}'.format(extra_reason)
        else:
            reason = 'Killed with {}'
        return ProcessResult(result_type=ProcessResult.KILLED,
                             reason=reason.format(signal_to_name(os.WTERMSIG(wait_status))))
    if os.WIFEXITED(wait_status):
        status_code = os.WEXITSTATUS(wait_status)
        if extra_reason:
            reason = '{}. Finished with {{}}'.format(extra_reason)
        else:
            reason = 'Finished with {}'
        success = not extra_reason and status_code == 0
        return ProcessResult(result_type=ProcessResult.FINISHED, status_code=os.WEXITSTATUS(wait_status),
                             reason=reason.format(os.WEXITSTATUS(wait_status)),
                             success=success)
    return None


SLEEP_INTERVAL = 0.5
TERMINATING_TIMEOUT = 0.5
BEFORE_KILL_TIMEOUT = 1


def run_process_with_timeout(*args, **kwargs):
    timeout = kwargs.get('timeout', 3600)

    proc = subprocess.Popen(list(args))
    start = time_time()
    while True:
        pid, wait_status = os.waitpid(proc.pid, os.WNOHANG)
        if pid and get_result(wait_status):
            return get_result(wait_status)

        if time_time() - start < timeout:
            time_sleep(SLEEP_INTERVAL)
        else:
            os.kill(proc.pid, signal.SIGTERM)
            time_sleep(TERMINATING_TIMEOUT)
            pid, wait_status = os.waitpid(proc.pid, os.WNOHANG)
            if pid and get_result(wait_status):
                return get_result(wait_status, 'Timed out killing with SIGTERM')

            time_sleep(BEFORE_KILL_TIMEOUT)

            os.kill(proc.pid, signal.SIGKILL)
            time_sleep(TERMINATING_TIMEOUT)
            pid, wait_status = os.waitpid(proc.pid, os.WNOHANG)
            if pid and get_result(wait_status):
                return get_result(wait_status, 'Timed out, didnt die from SIGTERM, killing with SIGKILL')
            return ProcessResult(result_type=ProcessResult.UNKNOWN,
                                 reason='SIGKILL was sent but the process didnt die instantly')


class ProcessError(Exception):
    pass


def process_job(*args, **kwargs):
    """
    :param args: run_process_with_timeout arguments
    :param kwargs: run_process_with_timeout keyword arguments
    """
    result = run_process_with_timeout(*args, **kwargs)

    if result.is_success():
        log.info('Process successfully finished %s', ' '.join(args))

    else:
        raise ProcessError('Process Error {} {}'.format(' '.join(args), result))
