# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

"""
Утилиты использования библиотек яндекса
"""

import json
import logging
import time
from datetime import datetime, timedelta
from hashlib import md5

import cookiemy
from django.conf import settings
from django.http import HttpResponseRedirect
from django.utils.http import urlencode
from six.moves.urllib_parse import unquote

from common.utils import request_helper
from common.utils.http import quick_urlopen
from common.utils.locations import langify


log = logging.getLogger(__name__)

# http://wiki.yandex-team.ru/MyCookie/%D0%9D%D0%BE%D0%BC%D0%B5%D1%80%D0%91%D0%BB%D0%BE%D0%BA%D0%B0
MY_LANG = 39


def my_cookie(request):
    """
    Возвращает объект управления cookie my

    см. http://wiki.yandex-team.ru/MyCookie
    см. http://wiki.yandex-team.ru/IljaGolubcov/libcookiemypython
    """

    if 'my' not in request.COOKIES:
        return None

    cookie = unquote(request.COOKIES['my'])

    setup = cookiemy.Setup()

    try:
        setup.parse(cookie)
    except RuntimeError:  # RuntimeError: Wrong cookie version при некорректной куки
        return None

    return setup


def lang_detect_find(req):
    url = settings.LANGDETECTSERVER_URL
    json_req = json.dumps(req)

    try:
        json_res = quick_urlopen(url, data=json_req).read()
    except Exception:
        log.warning('langdetect read failed', exc_info=True)

        return None

    try:
        res = json.loads(json_res)
    except ValueError:
        log.warning('langdetect json parse failed (%r)', json_res, exc_info=True)

        return None

    return res


def detect_lang(request, supported_langs=None, domain_lang=None):
    """
    Определяет язык страницы на основании настроек пользователя

    default_lang язык страницы по умолчанию
    supported_langs список возможных языков
    """

    domain_lang = domain_lang or request.domain_lang
    supported_langs = supported_langs or request.supported_langs
    my = my_cookie(request)

    if my:
        my_lang_code = my.find(MY_LANG)

        my_lang = str(my_lang_code[-1]) if my_lang_code else 0

    else:
        my_lang = 0

    req = {
        'language': request.META.get('HTTP_ACCEPT_LANGUAGE', ''),
        'cookie': my_lang,
        'domain': request.host,
        'filter': ','.join(supported_langs),
        'pass-language': request.yauser.lang,
        'default': domain_lang
    }

    res = lang_detect_find(req)

    if not res:
        return domain_lang

    lang = res['id']

    if lang not in supported_langs:
        lang = domain_lang

    return lang


def get_country_by_lang(lang):
    """
    Возвращает название страны по языку

    Пример:
    'Ua' == get_country_by_lang('uk')
    'Ru' == get_country_by_lang('ru')
    """

    ret = lang_detect_find({'default': lang})

    if ret:
        return ret['name']

    return lang.capitalize()


def make_secret_key(request):
    """
    Формирует секретный ключ

    см. http://wiki.yandex-team.ru/AlekseyPonomarev/secret-key
    """

    days = int(time.time() / 86400)

    if request.yauser.is_authenticated():
        return "u%s" % md5('%s::%s' % (request.yauser.uid, days)).hexdigest()

    elif 'yandexuid' in request.COOKIES:
        return "y%s" % md5('0:%s:%s' % (request.COOKIES['yandexuid'], days)).hexdigest()

    return None


def get_change_language_url(request, lang):
    params = {
        'intl': lang,
        'retpath': langify(request_helper.build_absolute_uri(request), lang),
        'sk': request.secret_key
    }

    return 'https://www.yandex.{tld}/portal/set/lang/?{query}'.format(tld=request.tld, query=urlencode(params))


def get_passport_host(request):
    return 'https://passport.yandex.{tld}'.format(tld=request.tld)


def get_passport_auth_url(request, short_service_name):
    """ https://doc.yandex-team.ru/Passport/passport-modes/reference/auth.xml """
    params = {
        'from': short_service_name,
        'retpath': request_helper.build_absolute_uri(request),
    }
    return '{host}/auth?{query}'.format(host=request.passport_host, query=urlencode(params))


def get_passport_exit_url(request):
    """ https://doc.yandex-team.ru/Passport/passport-modes/reference/logout.xml """
    params = {
        'mode': 'logout',
        'yu': getattr(request, 'yandexuid', ''),
        'retpath': request_helper.build_absolute_uri(request)
    }
    return '{host}/passport?{query}'.format(host=request.passport_host, query=urlencode(params))


def get_change_region_url(request):
    retpath_params = request.GET.copy()
    retpath_params['domredir'] = '1'
    retpath = request_helper.build_absolute_uri(request, request.path + '?' + retpath_params.urlencode())

    return 'https://www.yandex.{tld}/portal/set/region/?{query}'.format(tld=request.tld,
                                                                        query=urlencode({'retpath': retpath}))


def get_my_url(request, param, block, retpath=None):
    retpath = retpath or request_helper.build_absolute_uri(request)
    params = {
        'param': param,
        'block': block,
        'retpath': retpath,
        'sk': request.secret_key,
    }
    return 'https://www.yandex.{tld}/portal/set/my/?{query}'.format(tld=request.tld, query=urlencode(params))


def has_valid_yandexuid(request):
    """ https://wiki.yandex-team.ru/Cookies/yandexuid """

    if settings.HAS_VALID_YANDEXUID_ALWAYS_TRUE:
        return True

    # Срок жизни куки раньше был 10 лет, сейчас в связи с GDPR 2 вроде бы года, пока оставляем 10
    yandexuid_cookie_lifetime = 10

    yandexuid = request.yandexuid

    try:
        dt = datetime.fromtimestamp(float(yandexuid[-10:]))

    except Exception:
        is_valid_yandexuid = False

    else:
        now = request.naive_msk_now
        past = now - timedelta(days=yandexuid_cookie_lifetime*366)
        is_valid_yandexuid = past < dt < now + timedelta(1)

    if not is_valid_yandexuid:
        data = {
            'HTTP_HOST': request_helper.get_host(request),
            'REQUEST_URI': request.META.get('REQUEST_URI'),
            'HTTP_COOKIE': request.META.get('HTTP_COOKIE'),
            'REMOTE_ADDR': request_helper.get_ip(request),
            'HTTP_USER_AGENT': request.META.get('HTTP_USER_AGENT'),
        }

        log.warning('Invalid yandex uid %r', data)

    return is_valid_yandexuid


def get_auth_url(request):
    return 'https://passport.yandex.{tld}/auth/update/?{query}'.format(
        tld=request.tld,
        query=urlencode({'retpath': request_helper.build_absolute_uri(request)})
    )


def auth_response_redirect(request):
    redirect_url = get_auth_url(request)

    response = HttpResponseRedirect(redirect_url)
    response.set_cookie('Cookie_check', 'CheckCookieCheckCookie',
                        domain='.yandex.{tld}'.format(tld=request.tld))

    return response
