# coding: utf8
"""
Как подключить проект к Головану.
1) пишем письмо на rasp-admin "Прошу подключить группу(ы) rasp_test_{group}/rasp_{group} к Головану"
2) добавляем в deb пакет файл с конфигом для yasmagent, примерно такого содержания:
    instance="--instance-getter \"echo '$(hostname -f)/yasm_stats.json:5033@{itype} a_itype_{itype} a_prj_{your_prj}
    $GEO_YASM_CTYPE $GEO_YASM_GEO $GEO_YASM_TIER'\"" (только одной строкой, без переноса!)
   файл раскладываем так: install -D debian/$package.yasmagent $dest/etc/yandex/geo-yasm/$package.conf

Про значение тэгов itype, prj и тд читаем тут:
https://doc.yandex-team.ru/Search/golovan-quickstart/concepts/tags-golovan.html
"""

from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import time
from collections import namedtuple
from functools import wraps

import requests
import six
from django.conf import settings

from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from common.utils.dcutils import resource_explorer

log = logging.getLogger(__name__)

define_setting('YASMAGENT_PORT', {Configuration.DEVELOPMENT: '10029'}, default='11005')
define_setting('YASMAGENT_ITYPE', default=None)
define_setting('YASMAGENT_PROJECT', default=None)
define_setting('YASMAGENT_ENABLE_MEASURABLE', {
    Configuration.PRODUCTION: True,
    Configuration.TESTING: True
}, default=False)


@six.python_2_unicode_compatible
class YasmError(Exception):
    def __init__(self, error_code, error):
        self.error_code = error_code
        self.error = error

    def __str__(self):
        return '{}({}): {}'.format(self.__class__, self.error_code, self.error)


class Metric(namedtuple('Metric', ['name', 'value', 'suffix'])):
    """
    :parameter name: имя метрики
    :parameter value: значение
    :parameter suffix: общий суффикс
        https://doc.yandex-team.ru/Search/golovan-quickstart/concepts/signal-aggregation.html#sigopt-suffix
    """
    def to_dict(self, prefix=None):
        name = '{}_{}'.format(self.name, self.suffix)
        if prefix:
            name = '{}.{}'.format(prefix, name)
        return {
            'name': name,
            'val': self.value
        }


class YasmMetricSender(object):
    _GEO_TAG = None
    _DEFAULT_TIMEOUT = 0.1

    def __init__(self, tags=None, prefix=None):
        """

        :param tags: словарь tag: значение, возможные типы тагов тут:
              https://doc.yandex-team.ru/Search/golovan-quickstart/concepts/push.html
        :param prefix: общий префикс, например `billing` для метрик `billing.errors_count`, `billing.request_timings`
        """
        self._prefix = prefix
        self._tags = None
        self._additional_tags = tags
        self._url = 'http://localhost:{}'.format(settings.YASMAGENT_PORT)

    def send_one(self, metric, ttl=None):
        """
        Отправить одну метрику.
        :param metric: описание метрики
        :type metric: Metric
        :param ttl: время актуальности сигнала
        """
        data = metric.to_dict(self._prefix)
        data['tags'] = self.tags
        if ttl is not None:
            data['ttl'] = ttl
        self._send_to_yasm([data])

    def send_many(self, metrics, ttl=None):
        """
        Отправить несколько метрик пачкой.
        :param metrics: список отправляемых метрик
        :type metric: list[Metric]
        :param ttl: время актуальности сигнала
        """
        data = {
            'tags': self.tags,
            'values': [m.to_dict(self._prefix) for m in metrics]
        }
        if ttl is not None:
            data['ttl'] = ttl
        self._send_to_yasm([data])

    @property
    def tags(self):
        if self._tags is None:
            self._tags = self._build_tags()
            if self._additional_tags:
                self._tags.update(self._additional_tags)
        return self._tags

    def _send_to_yasm(self, data):
        log.info('Отправляем в yasm метрики: %s', data)
        response = requests.post(self._url, json=data, timeout=self._DEFAULT_TIMEOUT)
        if response.status_code in [200, 400]:
            response_data = response.json()
            if response_data['status'] != 'ok':
                raise YasmError(error_code=response_data['error_code'], error=response_data['error'])
        else:
            response.raise_for_status()

    @classmethod
    def _build_tags(cls):
        if not cls._GEO_TAG:
            cls._GEO_TAG = resource_explorer.get_current_dc()

        return {
            'itype': settings.YASMAGENT_ITYPE,
            'prj': settings.YASMAGENT_PROJECT,
            'ctype': settings.YANDEX_ENVIRONMENT_TYPE,
            'tier': settings.PKG_VERSION,
            'geo': cls._GEO_TAG
        }


class MeasurableDecoratorError(Exception):
    pass


DEFAULT_BUCKET_MULTIPLIER = 1.344597627113
DEFAULT_BUCKET_COUNT = 49
DEFAULT_FIRST_BUCKET_SIZE = 10


def get_buckets(first_bucket_size=DEFAULT_FIRST_BUCKET_SIZE, bucket_multiplier=DEFAULT_BUCKET_MULTIPLIER,
                bucket_count=DEFAULT_BUCKET_COUNT):
    result = [first_bucket_size]
    for i in range(bucket_count - 1):
        result.append(result[-1] * bucket_multiplier)
    return result


def get_bucket_values_generator(bucket_borders):
    def _generate_bucket_values(value):
        left_border, found_bucket = 0, False
        for right_border in bucket_borders:
            if not found_bucket and value < right_border:
                yield [left_border, 1]
                found_bucket = True
            else:
                yield [left_border, 0]
            left_border = right_border
    return _generate_bucket_values


class MeasurableDecorator(object):
    """
    Класс фабрика декораторов, собирающих метрики.
    Отправляет в Голован тайминги как гистограмму значений. Для того чтобы отправлять метрики по числу ошибок
    можно переопределить метод handle_error таким образом, чтобы он возвращал список метрик на основании полученного
    исключения.

    Пример использования:
        class measurable(MeasurableDecorator):
            default_prefix = 'trust'
            def _handle_error(self, ex):
                result = [Metric(self._name('errors_cnt'), 1, 'ammm')]
                metric_name = EXCEPTION_NAME_TO_METRIC[type(ex)]
                result.append(Metric(self._name(metric_name), 1, 'ammm'))
                return result

        @measurable()
        def func_to_decorate(arg):
            pass

    """
    prefix = None

    def __init__(self, endpoint_name=None, buckets=None):
        """
        :param endpoint_name: имя ендпойнта, если необходимо (по умолчанию имя эндпойнта берется из имени
            декорированной функции или метода)
        :param buckets: список правых границ бакетов
        """
        self.endpoint_name = endpoint_name
        self.func = None
        if buckets is None:
            buckets = get_buckets()
        self.bucket_values_generator = get_bucket_values_generator(buckets)

    def _name(self, metric_name):
        return '{}.{}'.format(self.endpoint_name, metric_name)

    def _handle_error(self, exc):
        return []

    def _send_metrics(self, metrics):
        try:
            YasmMetricSender(prefix=self.prefix).send_many(metrics)
        except Exception:
            log.exception('Ошибка при попытке отправить метрики')

    def __call__(self, func):
        if self.func is not None:
            raise MeasurableDecoratorError('Нельзя использовать один и тот же экземпляр декоратора дважды')

        self.func = func
        if self.endpoint_name is None:
            self.endpoint_name = func.__name__

        @wraps(func)
        def wrapper(*args, **kwargs):
            if not settings.YASMAGENT_ENABLE_MEASURABLE:
                return func(*args, **kwargs)
            started = time.time()
            metrics = []
            try:
                result = func(*args, **kwargs)
            except Exception as exc:
                metrics.extend(self._handle_error(exc))
                raise
            finally:
                completed_in = int((time.time() - started) * 1000)  # ms
                metrics.append(Metric(self._name('timings'), list(self.bucket_values_generator(completed_in)), 'ahhh'))
                self._send_metrics(metrics)
            return result
        return wrapper
