# -*- coding: utf-8 -*-

from django.conf import settings

from common.models.currency import Currency, Price
from common.models.geo import Settlement


class CurrencyInfo(object):
    def __init__(self, selected, set_preferred, country_base, src, rates, available, currencies):
        self.source = src
        self.base = Currency.BASE_CURRENCY
        self.country_base = country_base
        self.selected = selected

        self.formats = {}
        self.currencies = []
        self.rates = rates
        self.country_base = country_base

        for currency in currencies:
            self.formats[currency.code] = currency
            currency.rate = rates.get(currency.code)
            self.currencies.append(currency)

        self.order = [c.code for c in self.currencies]

        self.json = {
            'base': self.base,
            'country-base': country_base,
            'set-preferred': set_preferred,
            'selected': selected,
            'rates': rates,
            'order': self.order,
            'formats': {c.code: c.json() for c in currencies},
            'available': [code for code in self.order if code in available],
            'names': {c.code: {'name': c.L_name(), 'name_in': c.L_name_in()} for c in currencies}
        }

    def to_base(self, price):
        if not isinstance(price, Price):
            return price

        if price.currency not in self.rates:
            return None

        return price.value * self.rates[price.currency]

    def __json__(self):
        return self.json

    @classmethod
    def country_base_from_tld(cls, tld):
        country_base = 'RUR'
        if tld == 'com.tr':
            country_base = 'TRY'
        elif tld == 'ua':
            country_base = 'UAH'

        return country_base


def fetch_currency_info(request):
    currencies = list(Currency.get_ordered_queryset(request.NATIONAL_VERSION))

    geoid = request.GET.get('geo_id') or getattr(request, 'geoid', None) or settings.MOSCOW_GEO_ID

    src, rates = Currency.fetch_rates(currencies, geoid)

    # Оставляем только валюты с курсом и названием
    available = [
        currency
        for currency in currencies
        if (currency.L_name() and currency.L_name_in() and currency.code in rates)
    ]

    available = set(currency.code for currency in available)

    selected = None
    set_preferred = False

    selected_currency_code = request.GET.get('currency')

    if selected_currency_code in available:
        selected = selected_currency_code
        set_preferred = True

    # Пробуем определить по кукие
    if not selected:
        preferred = request.COOKIES.get('preferredcurrency')

        if preferred in available:
            selected = preferred

    # Если передан geoid, определяем по нему
    if not selected and request.GET.get('geo_id'):
        try:
            city = Settlement.objects.get(_geo_id=request.GET.get('geo_id'))

            if city.country:
                country_currency_id = city.country.currency_id

                for currency in currencies:
                    if currency.id == country_currency_id:
                        selected = currency.code

                        break

        except (Settlement.DoesNotExist, Settlement.MultipleObjectsReturned):
            pass

    # Пробуем определить по стране
    if not selected and request.client_city and request.client_city.country:
        country_currency_id = request.client_city.country.currency_id

        for currency in currencies:
            if currency.id == country_currency_id:
                selected = currency.code

                break

    # Пробуем определить по домену
    if not selected:
        try:
            selected = settings.DOMAIN_CURRENCY[request.tld]
        except KeyError:
            pass

    # Если все обломалось, используем рубли
    if not selected or selected not in available:
        selected = Currency.BASE_CURRENCY

    country_base = CurrencyInfo.country_base_from_tld(request.tld)
    return CurrencyInfo(
        selected,
        set_preferred,
        country_base,
        src,
        rates,
        available,
        currencies
    )


def get_currency_info(tld, currency=None):
    geo_id = settings.TLD_TO_GEO_ID[tld]
    currencies = list(Currency.objects.all())
    src, rates = Currency.fetch_rates(currencies, geo_id)

    # Оставляем только валюты с курсом и названием
    available = {
        currency.code
        for currency in currencies
        if (currency.L_name() and currency.L_name_in() and currency.code in rates)
    }

    set_preferred = False
    if currency in available:
        selected = currency
        set_preferred = True
    else:
        selected = settings.DOMAIN_CURRENCY.get(tld)

    # Если все обломалось, используем рубли
    if not selected or selected not in available:
        selected = Currency.BASE_CURRENCY

    country_base = CurrencyInfo.country_base_from_tld(tld)
    return CurrencyInfo(
        selected,
        set_preferred,
        country_base,
        src,
        rates,
        available,
        currencies
    )
