# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import datetime, timedelta
import logging
import warnings

import pytz
from django.db import connection

from common.apps.suburban_events.api import get_states_for_thread, EventStateType

from ..models.geo import Station, StationTerminal, StationMajority
from ..models.schedule import RTStation, Company, RThread
from ..models.transport import TransportModel, TransportType
from ..models_utils.i18n import RouteLTitle
from travel.rasp.library.python.common23.date import environment
from travel.rasp.library.python.common23.date.date import FuzzyDateTime, human_duration
from travel.rasp.library.python.common23.date.date_const import MSK_TZ
from travel.rasp.library.python.common23.date.run_mask import RunMask

from ..utils.exceptions import SimpleUnicodeException
from ..utils.warnings import RaspDeprecationWarning
from ..xgettext.i18n import xgettext


log = logging.getLogger(__name__)


EVENT_STATUS_OK = 'ok'
EVENT_STATUS_FACT_DELAY = 'fact-delay'
EVENT_STATUS_POSSIBLE_DELAY = 'possible-delay'


def get_near_date_future(year_days, today=None):
    warnings.warn('Use RunMask.first_run # 2015-04-14', RaspDeprecationWarning, stacklevel=2)
    today = today or environment.today()
    return RunMask.first_run(year_days, today)


def make_stops(thread, rtstations, start_dt, all_days):
    station_ids = [rts.station_id for rts in rtstations]
    stations = Station.objects.select_related('station_type', 'settlement', 'majority').in_bulk(station_ids)

    terminals = StationTerminal.objects.in_bulk([rts.terminal_id for rts in rtstations])

    t_model_ids = [rts.departure_t_model_id for rts in rtstations]
    t_models = TransportModel.objects.in_bulk(t_model_ids)

    stops = []
    naive_start_dt = start_dt.replace(tzinfo=None)

    if not all_days and thread.t_type_id == TransportType.SUBURBAN_ID:
        try:
            _thread_delay_state, event_states = get_states_for_thread(thread, naive_start_dt, rtstations)
        except Exception:
            log.exception(u'Незвестная ошибка при получении данных об опозданиях по нитке')
            event_states = {}
    else:
        event_states = {}

    for rts in rtstations:
        station = rts.station = stations[rts.station_id]
        city = station.settlement
        terminal = terminals.get(rts.terminal_id)

        t_model_id = rts.departure_t_model_id

        if t_model_id in t_models:
            segment_t_model = t_models[t_model_id]

        else:
            segment_t_model = thread.t_model

        stop_td = timedelta(0)

        if rts.tz_arrival is not None and rts.tz_departure is not None:
            stop_td = timedelta(minutes=rts.tz_departure - rts.tz_arrival)

        if rts.tz_arrival is None:
            arrival_dt = None

        else:
            arrival_dt = pytz.timezone(rts.time_zone).localize(
                naive_start_dt + timedelta(minutes=rts.tz_arrival)
            )

        if rts.tz_departure is None:
            departure_dt = None

        else:
            departure_dt = pytz.timezone(rts.time_zone).localize(
                naive_start_dt + timedelta(minutes=rts.tz_departure)
            )

        duration_td = arrival_dt and arrival_dt - start_dt

        local_arrival_dt = arrival_dt and arrival_dt.astimezone(station.pytz)
        local_departure_dt = departure_dt and departure_dt.astimezone(station.pytz)

        if rts.is_fuzzy:
            if local_arrival_dt and local_departure_dt:
                stop_time = local_departure_dt - local_arrival_dt

                local_departure_dt = FuzzyDateTime.fuzz(local_departure_dt)
                local_arrival_dt = FuzzyDateTime(local_departure_dt - stop_time)

            else:
                local_arrival_dt = local_arrival_dt and FuzzyDateTime.fuzz(local_arrival_dt)
                local_departure_dt = local_departure_dt and FuzzyDateTime.fuzz(local_departure_dt)

        stop_dict = {
            'title': station.L_title() or '',
            'id': station.id,
            'city': city,
            'station': station,
            'terminal': terminal,
            'departure_t_model': segment_t_model,
            'city_size': city and city.city_size() or '',
            'majority': station.majority_id or StationMajority.IN_TABLO_ID,
            'arrival': local_arrival_dt,
            'departure': local_departure_dt,
            'duration': duration_td,
            'stop_time': stop_td,
            'thread_number': thread.number,
            'majority_code': station.majority.code,
            'is_technical_stop': rts.is_technical_stop,
            'L_platform': rts.L_platform,
            'l_platform': rts.L_platform(),
            'is_fuzzy': rts.is_fuzzy,
            'is_combined': rts.is_combined,
            'rtstation': rts,
            'event_states': make_event_states(local_departure_dt, local_arrival_dt, event_states.get(rts))
        }

        stops.append(stop_dict)

    return stops


def make_event_states(local_departure_dt, local_arrival_dt, event_states):
    if not event_states:
        return None

    nonstop = local_departure_dt == local_arrival_dt

    # TODO: для безостановочных станций время в нашей БД сильно некорректное.
    # Поэтому опоздание будет показано так же некорректно.
    # В будущем научимся показывать более правильно время опоздания по безостановочным станциям.
    if nonstop:
        return None

    result = {
        'message': None,
        'status': None
    }
    for event in ('arrival', 'departure'):
        event_state = getattr(event_states, event)
        if event_state:
            local_dt = local_departure_dt if event == 'departure' else local_arrival_dt
            minutes_from, minutes_to = get_event_state_minutes(local_dt, event_state)

            status = get_event_status(event_state.type, minutes_to)
            if not status:
                continue
            message = get_event_message(event, minutes_from, minutes_to, status, nonstop)
            result['status'] = status
            result['message'] = message
            result[event] = {
                'status': status,
                'message': message,
                'minutes_from': None if status == EVENT_STATUS_OK else minutes_from,
                'minutes_to': None if status == EVENT_STATUS_OK else minutes_to
            }

    return result


def get_event_state_minutes(local_dt, event_state):
    # TODO: Сейчас в event_state есть атрибуты minutes_from, minutes_to. Но там лежат некорректные значения.
    # Когда значения этих атрибутов будут пофиксены, нужно будет брать их.
    # Сейчас пока костыль с использованием атрибута dt.
    if not event_state.dt:
        return None, None
    event_state_dt_aware = pytz.timezone(event_state.tz).localize(event_state.dt)
    minutes = (event_state_dt_aware - local_dt).total_seconds() // 60
    return minutes, minutes


def get_event_message(event, minutes_from, minutes_to, status, nonstop):
    if status == EVENT_STATUS_OK:
        return xgettext(u'по расписанию')
    if status == EVENT_STATUS_POSSIBLE_DELAY:
        if minutes_to:
            if minutes_from != minutes_to:
                return xgettext(u'возможно опоздание на <minutes_from /> \N{em dash} <minutes_to /> мин.',
                                minutes_from=minutes_from, minutes_to=minutes_to)
            return xgettext(u'возможно опоздание на <minutes /> мин.', minutes=minutes_to)
        return xgettext(u'возможно опоздание')
    if status == EVENT_STATUS_FACT_DELAY:
        if nonstop:
            return xgettext(u'опаздывает на <minutes /> мин.', minutes=minutes_to)
        if event == 'departure':
            if minutes_to:
                return xgettext(u'отправился с опозданием на <minutes /> мин.', minutes=minutes_to)
            return xgettext(u'отправился с опозданием')
        if event == 'arrival':
            if minutes_to:
                return xgettext(u'прибыл с опозданием на <minutes /> мин.', minutes=minutes_to)
            return xgettext(u'прибыл с опозданием')

    return None


def get_event_status(event_state_type, minutes):
    if event_state_type == EventStateType.POSSIBLE_DELAY:
        if minutes is None or minutes > 1:
            return EVENT_STATUS_POSSIBLE_DELAY
        return 'ok'
    if event_state_type == EventStateType.FACT:
        if (minutes or 0) > 1:
            return EVENT_STATUS_FACT_DELAY
        return 'ok'


def get_thread_departure_date(requested_departure_date, thread):
    if (requested_departure_date and
            isinstance(requested_departure_date, datetime)):
        requested_departure_date = requested_departure_date.date()

    near_departure_date = get_near_date_future(thread.year_days)

    departure_date = requested_departure_date or near_departure_date
    if not departure_date:
        raise ThreadWithoutDaysError(u"rthread %s has no start date" % thread.id)

    return departure_date


class ThreadDataError(SimpleUnicodeException):
    pass


class ThreadWithoutStationsError(ThreadDataError):
    u""" Нить меньше чем с двумя станциями """
    pass


class ThreadWithoutDaysError(ThreadDataError):
    u""" Нить без дней хождения """
    pass


def get_interval_string(thread):
    if thread.density:
        return thread.density

    if not thread.period_int:
        return u''

    duration = timedelta(minutes=thread.period_int)

    return xgettext(u'интервал: <interval_duration />', interval_duration=human_duration(duration))


def add_merge_info(thread_conext, stops, route):
    # Ищем есть ли merged flights на этот день
    cursor = connection.cursor()

    query = """SELECT id, merged_flight_id FROM z_tablo2
        WHERE DATE(direction)=%(direction)s AND merged_flight_id IS NOT NULL
        AND route_id=%(route_id)s
        AND station_id=%(station_id)s
        """
    row = None
    for s in stops:
        departure = s.get('departure')
        if departure:
            cursor.execute(
                query.replace('direction', 'departure'),
                {
                    'departure': departure.date(),
                    'route_id': route.id,
                    'station_id': s['id']
                })

            row = cursor.fetchone()
            if row:
                break

        arrival = s.get('arrival')
        if arrival:
            cursor.execute(
                query.replace('direction', 'arrival'),
                {
                    'arrival': arrival.date(),
                    'route_id': route.id,
                    'station_id': s['id']
                })

            row = cursor.fetchone()
            if row:
                break

    if row is not None and row[1] is not None:
        cursor.execute("SELECT r_number, thread_id, company_id FROM z_tablo2 WHERE id=%s", [row[1]])

        row = cursor.fetchone()

        if row:
            number, thread_id, company_id = row

            if thread_id is not None:
                thread = RThread.objects.get(id=thread_id)
                thread_conext.update({'merged_thread': thread,
                                      'merged_number': thread.number})
            else:
                thread_conext.update({'merged_thread': None,
                                      'merged_number': number})

            if company_id:
                try:
                    thread_conext['merged_company'] = Company.hidden_manager.get(pk=company_id)
                except Company.DoesNotExist:
                    pass


def get_thread_context(thread, departure_date, all_days):
    """ Получает данные о нитке
        Вход: нитка, дата отправления
        Выход: словарь
    """
    now_aware = environment.now_aware()

    query_type = 'thread'

    route = thread.route

    # RASP-3968
    if route.hidden:
        raise ThreadDataError(u"Route is hidden")

    if thread.type.code == 'cancel':
        raise ThreadDataError(u'Thread is cancelled')

    route_threads = list(route.rthread_set.all().select_related())

    t_model = thread.t_model

    tz_start_time = thread.tz_start_time
    start_date = departure_date

    RouteLTitle.fetch([obj.L_title for obj in [thread] + route_threads])

    thread_context = {
        'thread_number': thread.number,
        'thread_uid': thread.uid,
        'thread_id': thread.id,
        'thread': thread,
        'title': thread.title,
        'l-title': thread.L_title(),
        'transportType': thread.t_type_id,
        't_type': thread.t_type,
        'type': query_type,
        'route_threads': route_threads,
        't_model': t_model,
        'is_interval': thread.is_interval,
        'begin_time': thread.begin_time,
        'end_time': thread.end_time,
        'density': get_interval_string(thread),
        'comment': thread.comment,
    }

    rtstations = list(RTStation.objects.filter(thread__id=thread.id).order_by('id'))
    # thread_id, arrival, departure, day_shift, station_id, terminal_id, departure_t_model_id,
    # is_technical_stop, platform, is_fuzzy, is_combined

    start_dt = thread.pytz.localize(datetime.combine(start_date, tz_start_time))

    thread_context['start_datetime'] = start_dt

    thread_context['one_t_model'] = True

    stops = make_stops(thread, rtstations, start_dt, all_days)

    t_model = thread.t_model
    for stop in stops:
        if stop['departure_t_model'] != t_model:
            # Показываем по сегментам
            thread_context['one_t_model'] = False
            break

    current_station_id = None
    for stop in stops:
        if stop['arrival'] is not None and stop['arrival'] < now_aware:
            current_station_id = stop['station'].id

    # В итоге, если рейс уже ушел будет совпадать с последней стацией,
    # Возможно это неправильно, но так было до рефакторинга
    thread_context['currentStation'] = current_station_id

    if len(stops) < 2:
        raise ThreadWithoutStationsError(u"Слишком мало станций у нитки '%s'" % thread.id)

    if thread.t_type_id == TransportType.PLANE_ID:
        add_merge_info(thread_context, stops, route)

    stops[0]['arrival'] = None
    stops[-1]['departure'] = None

    thread_context['departure'] = start_dt.astimezone(MSK_TZ)
    thread_context['stops'] = stops

    # FIXME: для совместимости с мобильной версией, т.к. там жуткий код
    thread_context['stations'] = stops

    return thread_context


def calc_thread_start_date(thread, station_from, naive_departure_dt):
    """
    :type thread: RThread
    :type station_from: Station
    :param naive_departure_dt: время отправления со станции во временной зоне станции
    :type naive_departure_dt: date
    :return date | None
    """
    rtstations = list(RTStation.objects.filter(
        thread__id=thread.id,
        station__id=station_from.id).order_by('id')
    )
    if len(rtstations) > 1:
        log.warning(u"Много rtstations ({number}) для нитки {thread_uid} и станции {st_id}".format(
            number=len(rtstations),
            thread_uid=thread.uid,
            st_id=station_from.id
        ))

    departure_dt = station_from.pytz.localize(naive_departure_dt)
    for rtstation in rtstations:
        expected_start_date = rtstation.calc_thread_start_date(
            event_date=naive_departure_dt.date(),
            event_tz=station_from.pytz
        )
        if not thread.runs_at(expected_start_date):
            continue
        expected_departure = rtstation.get_departure_dt(
            datetime.combine(expected_start_date, thread.tz_start_time), out_tz=station_from.pytz
        )
        if expected_departure == departure_dt:
            return expected_start_date

    log.warning(u"Не нашлось даты старта нитки для: {thread_uid}, {station_id}, {departure}".format(
        thread_uid=thread.uid,
        station_id=station_from.id,
        departure=naive_departure_dt
    ))
    return None
