# coding: utf-8

from datetime import datetime

from common.models.schedule import RTStation
from common.models.transport import TransportType
from common.utils.iterrecipes import unique_everseen


def extend_with_stations(threads, now_aware):
    # Выбираем первые и последние станции
    first_stations = {}
    last_stations = {}

    threads_by_id = {thread.id: thread for thread in threads}

    # Промежуточные станции
    rtstations = list(
        RTStation.objects
                 .filter(thread__in=threads)
                 .exclude(tz_departure__isnull=False, tz_arrival__isnull=False)
                 .select_related('station')
    )

    for rts in rtstations:
        thread = threads_by_id[rts.thread_id]

        thread_tz_today = now_aware.astimezone(thread.pytz).date()

        first_run = thread.first_run(thread_tz_today)

        if first_run is None:
            continue

        naive_start_dt = datetime.combine(first_run, thread.tz_start_time)

        station = rts.station

        if rts.tz_arrival is None:
            # TODO: нужно будет считать время и дни хождения по жд-времени страны для поездов
            # и по местному для самолётов

            departure_dt = rts.get_departure_loc_dt(naive_start_dt)
            station.departure_time = departure_dt

            shift = (departure_dt.date() - naive_start_dt.date()).days

            # Дни хождения
            station.schedule = thread.L_days_text(
                shift,
                except_separator=", ",
                html=False,
                thread_start_date=first_run
            )

            first_stations[rts.thread_id] = station

        if rts.tz_departure is None:
            station.arrival_time = rts.get_arrival_loc_dt(naive_start_dt)
            last_stations[rts.thread_id] = station

    for thread in threads:
        try:
            thread.first_station = first_stations[thread.id]
            thread.last_station = last_stations[thread.id]
        except KeyError:
            pass
        else:
            yield thread


def process_threads(threads, now_aware):
    # Убираем дубли
    threads = list(unique_everseen(threads, key=lambda t: (t.number, t.title)))

    ttypes = TransportType.objects.all().in_bulk(set(t.route.t_type_id for t in threads))

    for t in threads:
        t.t_type_id = t.route.t_type_id
        t.t_type = ttypes[t.t_type_id]

    return sorted(
        extend_with_stations(threads, now_aware),
        key=lambda thread: (thread.t_type_id,
                            thread.title or thread.route.title)
    )
