# -*- coding: utf-8 -*-

from datetime import datetime, time, timedelta

from common.models.geo import Settlement
from common.xgettext.i18n import gettext, mark_gettext, dynamic_gettext


RAILWAY_TZ_NAMES = {
    Settlement.MOSCOW_ID: mark_gettext(u'московское время'),
    143: mark_gettext(u'киевское время'),
    10318: mark_gettext(u'время Душанбе'),
}


class TimezoneChoice(object):
    def __init__(self, city, name, shifts):
        self.city = city
        self.name = name
        self.shifts = shifts

        self.city_id = None if city is None else city.id


def fill_tz_context(request, context, cities, dates=None, countries=None):
    """
    Функция, добавляющая в контекст параметры, необходимые для работы переключателя времени.
    Параметры:
    cities - список актуальных объектов Settlement

    Список актуальных дат:
    dates - просто список объектов date
    """
    dates = [min(dates) - timedelta(days=1), max(dates) + timedelta(days=7)]

    tz_countries = set(c.country for c in cities)

    if countries:
        tz_countries.update(countries)

    tz_countries.discard(None)

    railway_tz_cities = set(c.get_railway_tz_city() for c in tz_countries)

    railway_tz_cities.discard(None)

    # Убираем неподдерживаемые
    railway_tz_cities = set(c for c in railway_tz_cities if c.id in RAILWAY_TZ_NAMES)

    tz_choices = []

    # RASP-6253, добавляем местное время только если присутствует ЖД-время разных
    # стран или временная зона переданных городов отличается от ЖД-времени
    all_tz_choices = set(c.time_zone for c in railway_tz_cities.union(cities))

    if not railway_tz_cities or len(all_tz_choices) > 1:
        tz_choices.append(TimezoneChoice(None, gettext(u'местное время'), None))

    shifts = set()

    tz_cities = []

    for city in railway_tz_cities:
        city_shifts = tuple(city.get_time_zone(msk_datetime=datetime.combine(d, time(0, 0))) for d in dates)

        if city_shifts not in shifts:
            name = dynamic_gettext(RAILWAY_TZ_NAMES[city.id])

            tz_choices.append(TimezoneChoice(city, name, city_shifts))
            tz_cities.append(city)

            shifts.add(city_shifts)

    context['timezones'] = [(choice.city_id, choice.name, choice.shifts) for choice in tz_choices]
    context['tz_cities'] = tz_cities
    context['tz_names'] = dict((choice.city_id, choice.name) for choice in tz_choices)

    selected = tz_choices[0]

    try:
        requested = int(request.GET['time_zone'])
    except (KeyError, ValueError):
        requested = None

    if requested:
        for choice in tz_choices:
            if choice.city_id == requested:
                selected = choice
                break

    context['time_zone'] = selected.city
    context['time_zone_name'] = selected.name

    return context
