#!/usr/bin/python2.7
# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import glob
import os
import shutil
from argparse import ArgumentParser
from collections import defaultdict, namedtuple
from subprocess import check_call


OVERRIDE_TAG = '@override'
RequirementSpec = namedtuple('RequirementSpec', ['package', 'version', 'override'])


def read_requirements(path):
    requirements = []

    print('>> read requirements files for rule: [{}]'.format(path))
    for path in glob.glob(path):
        print('>>>> read requirements file: [{}]'.format(path))
        with open(path) as f:
            requirements += f.read().decode('utf-8').splitlines()

    return requirements


def _parse_requirement(line):
    splitted_line = line.split('==')
    if len(splitted_line) == 1:
        return RequirementSpec(splitted_line[0], None, False)
    if len(splitted_line) > 2:
        splitted_line = [splitted_line[0], '=='.join(splitted_line[1:])]
    package, rest = splitted_line
    if OVERRIDE_TAG in rest:
        return RequirementSpec(package, rest, True)
    return RequirementSpec(package, rest, False)


def _choose_spec(specs):
    for spec in specs:
        if spec.override:
            break
    return spec


def _format_spec(spec):
    if spec.version:
        return '{spec.package}=={spec.version}'.format(spec=spec)
    return spec.package


def normalize(requirements):
    specs_by_packages = defaultdict(list)
    for r in set(requirements):
        if not r:
            continue
        spec = _parse_requirement(r)
        specs_by_packages[spec.package.lower()].append(spec)
    return [
        _format_spec(_choose_spec(specs))
        for specs in sorted(specs_by_packages.values())
    ]


def save_requirements(name, requirements):
    with open(name, 'w') as f:
        normalized_requirements = normalize(requirements)
        f.writelines([
            '{}\n'.format(r).encode('utf-8') for r in normalized_requirements
        ])

        print('>> save requirements: {}'.format(name))
        for r in normalized_requirements:
            print('>>>> *) {}'.format(r).encode('utf-8'))


def join_requirements(is_dev, py3):
    print("Start join requirements config")

    requirements = read_requirements("../*/requirements.txt")
    if py3:
        requirements += read_requirements("../*/requirements_py3.txt")
    else:
        requirements += read_requirements("../*/requirements_py2.txt")
    dev_requirements = []
    if is_dev:
        dev_requirements = read_requirements("../*/dev_requirements.txt")
        if py3:
            dev_requirements += read_requirements("../*/dev_requirements_py3.txt")
        else:
            dev_requirements += read_requirements("../*/dev_requirements_py2.txt")
    requirements_nodeps = read_requirements("../*/requirements_nodeps.txt")

    save_requirements('freeze_requirements.txt', requirements + dev_requirements)
    save_requirements('freeze_requirements_nodeps.txt', requirements_nodeps)

    print("Finish join requirements config")


def cmd(command):
    print('\n\n>>>> {}\n\n'.format(command))
    check_call(command.split(' '))


def install_py2():
    print("\n\nInstall requirements")

    if os.path.exists('virtualenv'):
        shutil.rmtree('virtualenv')
    cmd('virtualenv --python=python2.7 virtualenv --system-site-packages --no-setuptools')

    # Иногда получается очень длинный shebang в pip и он обрубается, поэтому лучше
    # запускать так

    cmd('virtualenv/bin/python -m pip install --upgrade pip~=9.0 setuptools~=37.0')
    cmd('virtualenv/bin/python -m pip install -I -r freeze_requirements.txt')
    cmd('virtualenv/bin/python -m pip install -I --no-deps -r freeze_requirements_nodeps.txt')

    print("Finish install requirements")


def install_py3():
    print("\n\nInstall requirements")

    if os.path.exists('virtualenv_py3'):
        shutil.rmtree('virtualenv_py3')
    cmd('virtualenv --python=python3.7 virtualenv_py3 --system-site-packages')

    # Иногда получается очень длинный shebang в pip и он обрубается, поэтому лучше
    # запускать так

    cmd('virtualenv_py3/bin/python -m pip install --upgrade pip~=9.0 setuptools~=37.0')
    cmd('virtualenv_py3/bin/python -m pip install -I -r freeze_requirements.txt')
    cmd('virtualenv_py3/bin/python -m pip install -I --no-deps -r freeze_requirements_nodeps.txt')

    print("Finish install requirements")


def clear():
    print("\n\nStart clear requirements files")
    os.remove('freeze_requirements.txt')
    os.remove('freeze_requirements_nodeps.txt')
    print("Finish clear install requirements")


if __name__ == '__main__':
    parser = ArgumentParser()
    parser.add_argument("--dev", dest="is_dev", default=False,
                        action="store_true", help="fetch development requirements configs")
    parser.add_argument("--skip-join",
                        action="store_true", dest="skip_join", default=False,
                        help="don't rejoin requirements configs")
    parser.add_argument("--skip-install",
                        action="store_false", dest="install_requirements", default=True,
                        help="don't install requirements")
    parser.add_argument("--keep",
                        action="store_true", dest="keep_requirements", default=False,
                        help="don't remove joined requirements configs")
    parser.add_argument("--py3", dest="py3", default=False,
                        action="store_true", help="setup virtualenv_py3 with python3 inside")
    args = parser.parse_args()

    os.chdir(os.path.realpath(os.path.join(os.path.realpath(__file__), '..')))

    if not args.skip_join:
        join_requirements(is_dev=args.is_dev, py3=args.py3)
    if args.install_requirements:
        if args.py3:
            install_py3()
        else:
            install_py2()
    if not args.keep_requirements:
        clear()
