# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import contextlib
import logging
import os
import socket
import threading
import uuid
from datetime import timedelta

from gevent.monkey import is_module_patched
from greenlet import getcurrent
from pymongo import ReturnDocument, WriteConcern

from travel.rasp.library.python.common23.date.environment import now_utc
from common.workflow.errors import UnableToUpdate, CantGetLock, CantReleaseLock
from common.workflow.utils import get_by_dotted_path, document_from_mongo_result

nothing = object()  # value to fill default arguments

log = logging.getLogger(__name__)


class DocumentLocker(object):
    def __init__(self, document, namespace, lock_alive_time=20, lock_update_interval=5):
        # document attributes
        self.document = document
        self.namespace = namespace

        # lock attributes
        self.lock_uid = '{}__{}__{}'.format(socket.getfqdn(), os.getpid(), str(uuid.uuid4()))
        self.lock_alive_time = lock_alive_time
        self.lock_update_interval = lock_update_interval
        self._lock_modified = self.store and self.store.get('lock_modified')
        self._lock_modified_lock = threading.Lock()

        self.lock_uid_key = '{}.lock_uid'.format(self.namespace)
        self.lock_modified_key = '{}.lock_modified'.format(self.namespace)

        self._stopped = threading.Event()
        self._updater = threading.Thread(target=self._update_lock)
        # позволяет отслеживать, когда происходит обновление блокировки
        self.update_event = threading.Event()

        self._is_lock_acquired = False

    @property
    def store(self):
        return get_by_dotted_path(self.document, self.namespace, None)

    @property
    def collection(self):
        return self.document.__class__._get_collection().with_options(write_concern=WriteConcern(j=True, w='majority'))

    @contextlib.contextmanager
    def __call__(self):
        if not self._acquire_lock():
            raise CantGetLock("Can't get lock {} for document {}".format(self.lock_uid, self.document.id))

        try:
            self._updater.start()
            yield
        finally:
            self._stopped.set()
            if self._updater.is_alive():
                self._updater.join()
            self._release_lock()

    def build_lock_query(self):
        return {
            '_id': self.document.id,
            self.lock_uid_key: self.lock_uid
        }

    def update_document_raw(self, raw_update, **kwargs):
        kwargs['return_document'] = ReturnDocument.AFTER
        result = self.collection.find_one_and_update(self.build_lock_query(), raw_update, **kwargs)

        if not result:
            # Something is terribly wrong. Either document doesn't exist or somebody took our lock.
            raise UnableToUpdate('Unable to update document {} with lock {}'.format(self.document.id, self.lock_uid))

        self.document = document_from_mongo_result(self.document.__class__, result)

        return self.document

    @property
    def lock_modified(self):
        with self._lock_modified_lock:
            return self._lock_modified

    @lock_modified.setter
    def lock_modified(self, value):
        with self._lock_modified_lock:
            self._lock_modified = value

    def is_up_to_date(self):
        return (now_utc() - self.lock_modified).total_seconds() < self.lock_alive_time

    def is_locked(self):
        """Locked by me right now"""
        return bool(self._is_lock_acquired and self._updater.is_alive() and self.is_up_to_date())

    def _build_lock_set_dict(self, lock_value=nothing):
        if lock_value == nothing:
            lock_value = self.lock_uid
        last_modifed = now_utc()
        return {
            '$set': {
                self.lock_uid_key: lock_value,
                self.lock_modified_key: last_modifed,
            }
        }, last_modifed

    def _acquire_lock(self):
        """ Пытаемся получить себе лок. Это можно сделать если он не взят либо просрочен. """
        log.debug(u'Acquire lock {} for document {}'.format(self.lock_uid, self.document.id))

        # забираем лок либо по None, либо по значению просроченного лока
        lock_set_dict, lock_modified = self._build_lock_set_dict()
        result = self.collection.find_one_and_update({
            '_id': self.document.id,
            '$or': [
                {self.lock_uid_key: None},
                {self.lock_uid_key: {'$exists': False}},
                {
                    self.lock_modified_key: {
                        '$lt': now_utc() - timedelta(seconds=self.lock_alive_time)
                    }
                }
            ]
        }, lock_set_dict, projection={})
        if result:
            self.lock_modified = lock_modified
            self._is_lock_acquired = True
            return True

        log.debug(u"Can't acquire lock {} for document {}".format(self.lock_uid, self.document.id))
        return False

    def _update_lock(self):
        while True:
            if self._stopped.wait(self.lock_update_interval):
                return

            log.debug(u'Update lock {} for document {}. Thread/greenlet: {}'.format(
                self.lock_uid, self.document.id, self._current_thread()
            ))
            lock_set_dict, lock_modified = self._build_lock_set_dict()
            result = self.collection.find_one_and_update(self.build_lock_query(), lock_set_dict)
            if not result:
                msg = "Cant get lock {} in locker for document {}".format(self.lock_uid, self.document.id)
                log.error(msg)
                raise CantGetLock(msg)

            self.lock_modified = lock_modified
            self.update_event.set()
            self.update_event.clear()

    def _release_lock(self):
        log.debug('Release lock {} for document {}'.format(self.lock_uid, self.document.id))
        lock_set_dict, lock_modified = self._build_lock_set_dict(None)
        result = self.collection.find_one_and_update(self.build_lock_query(), lock_set_dict,
                                                     return_document=ReturnDocument.AFTER)
        if not result:
            raise CantReleaseLock("Can't release lock {} for document {}".format(self.lock_uid, self.document.id))

        self.document = document_from_mongo_result(self.document.__class__, result)

        self.lock_modified = lock_modified
        self._is_lock_acquired = False

    @staticmethod
    def _current_thread():
        if is_module_patched('threading'):
            return getcurrent()
        return threading.current_thread()
