# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import os
import socket
import uuid
from datetime import timedelta

from bson import ObjectId
from mongoengine.queryset import transform
from pymongo import ReturnDocument
from pymongo.errors import ConnectionFailure
from pymongo.write_concern import DEFAULT_WRITE_CONCERN
from ylog.context import log_context

from common.db.switcher import switcher
from travel.rasp.library.python.common23.date.environment import now_utc
from common.utils.namedtuple import namedtuple_with_defaults
from common.utils.try_hard import try_hard
from common.workflow.errors import CantGetLock
from common.workflow.locker import DocumentLocker
from common.workflow.utils import (
    get_by_dotted_path, get_update_with_prefix, merge_updates, set_by_dotted_path, document_from_mongo_result
)

log = logging.getLogger(__name__)
history_log = logging.getLogger('common.workflow.history')

DEFAULT_NAMESPACE = 'process'
DEFAULT_MAX_RETRIES = 5
DEFAULT_MAX_PROCESS_LIFE_TIME = timedelta(minutes=3)


class ActionResult(object):
    def __init__(self, result, doc_update=None, wait_till=None, params=None):
        self.result = result
        self.doc_update = doc_update
        self.wait_till = wait_till
        self.params = params


class ProcessEvent(namedtuple_with_defaults(
    'ProcessEvent',
    ['name', 'params', 'doc_update', 'is_external', 'uid', 'wait_till', 'created_at'],
    defaults={'is_external': True}
)):
    def __new__(cls, *args, **kwargs):
        kwargs['created_at'] = now_utc()
        if not ('uid' in cls._fields[:len(args)] or 'uid' in kwargs):
            kwargs['uid'] = ObjectId()
        return super(ProcessEvent, cls).__new__(cls, *args, **kwargs)

    @classmethod
    def parse_from_action_result(cls, action_result):
        if isinstance(action_result, tuple):
            return ProcessEvent(action_result[0], doc_update=action_result[1], is_external=False)
        elif isinstance(action_result, ActionResult):
            return ProcessEvent(action_result.result, doc_update=action_result.doc_update,
                                wait_till=action_result.wait_till, is_external=False, params=action_result.params)
        else:
            return ProcessEvent(action_result, is_external=False)


class Process(object):
    EXCEPTION_STATE = 'unhandled_exception_state'
    NEED_RESCHEDULE_STATE = 'need_reschedule_state'
    EXCEPTION_EVENT = ProcessEvent('unhandled_exception', is_external=False)
    raise_on_error = False

    def __init__(self, scheme, document, namespace=DEFAULT_NAMESPACE, lock_uid=None,
                 max_retries=DEFAULT_MAX_RETRIES, document_locker_class=DocumentLocker,
                 max_lifetime=DEFAULT_MAX_PROCESS_LIFE_TIME):
        """
        :param scheme: workflow.Scheme
        :param document: mongoengine.Document instance
        :param namespace: path to process data on document. Path shouldn't contain mongo arrays
        :param lock_uid: running process lock identificator
        :param max_retries: number of retries in case of mongo ConnectionFailure
        :param max_lifetime: timedelta, after that process will be restarted
        """
        self.scheme = scheme
        self.namespace = namespace
        if not lock_uid:
            lock_uid = '{}__{}__{}'.format(socket.getfqdn(), os.getpid(), str(uuid.uuid4()))
        self.lock_uid = lock_uid

        if max_retries is not None:
            self.update = try_hard(max_retries=max_retries, retriable_exceptions=(ConnectionFailure,))(self._update)
        else:
            self.update = self._update

        self.document_locker = document_locker_class(document, namespace,
                                                     lock_alive_time=scheme['lock_alive_time'],
                                                     lock_update_interval=scheme['lock_update_interval'])
        self.max_lifetime = max_lifetime

    @property
    def document(self):
        return self.document_locker.document

    @document.setter
    def document(self, document):
        self.document_locker.document = document

    @property
    def collection(self):
        return self.document_locker.collection

    @property
    def process(self):
        return get_by_dotted_path(self.document, self.namespace, None)

    @property
    def state(self):
        return self.process['state']

    @property
    def state_config(self):
        if self.state == self.EXCEPTION_STATE:
            return {}

        return self.scheme['states'][self.state]

    @property
    def external_event(self):
        return self.process.get('external_event')

    @property
    def terminal_states(self):
        """
        Возвращаем терминальные стейты - те, в которых нет кода для выполнения.
        Из таких состояний можной выйти только через send_external_event.
        """

        states = [state for state, state_conf in self.scheme['states'].items() if not state_conf.get('do')]

        # если не задано специальное описание для ошибочного стейта, считаем его терминальным
        if self.EXCEPTION_STATE not in self.scheme['states']:
            states.append(self.EXCEPTION_STATE)

        return states

    def _get_transition_state(self, event):
        if event == self.EXCEPTION_EVENT:
            return self.EXCEPTION_STATE

        if self.state == self.EXCEPTION_STATE:
            history_log.warning('Попытка взять transition_state у EXCEPTION_STATE. '
                                'state={}, scheme={} document={} process={}'.format(self.state, self.scheme,
                                                                                    self.document, self.process))
            return self.EXCEPTION_STATE

        return self.state_config['transitions'].get(event.name)

    def get_namespace(self, namespace):
        return get_by_dotted_path(self.document, self.namespace + '.' + namespace)

    def init_process(self):
        if self.process is None:
            set_by_dotted_path(self.document, self.namespace, {})
        process_update__set = {}
        if not self.process.get('state'):
            process_update__set['state'] = self.scheme['initial_state']
        if self.process.get('data') is None:
            process_update__set['data'] = {}
        if self.process.get('external_events') is None:
            process_update__set['external_events'] = []
        if self.process.get('history') is None:
            process_update__set['history'] = []
        if process_update__set:
            self.update({'$set': process_update__set})

    def run(self, log_context_data=None):
        """
        :param log_context_data - контекст логирования
        """
        with log_context(**(log_context_data or {})):
            try:
                with self.document_locker():
                    return self._actual_run()
            except CantGetLock:
                log.warning('Не смогли получить лок для %s', self.document.id)

    def _actual_run(self):
        """ Исполняем процесс по шагам по заданной схеме. """
        restart_at = now_utc() + self.max_lifetime
        self.init_process()
        try:
            while True:
                is_last_state = self._process_state()
                if is_last_state:
                    break
                # рестарт процесса https://st.yandex-team.ru/TRAINS-308#1535979636000
                if restart_at < now_utc():
                    return Process.NEED_RESCHEDULE_STATE

        except Exception:
            history_log.exception('Unhandled exception in run')
            raise

        return self.state

    def _process_state(self):
        """
        :return: Закончить ли обработку процесса
        """
        if not self.document_locker.is_locked():
            history_log.info('Lock is staled.')
            return True

        external_event = self._get_next_processable_event()
        if external_event:
            self._receive_event(external_event)
        else:
            # Если нечего запускать - это завершающий стейт, и все события уже были обработаны
            # Из него можно выйти только через external_event.
            if self.state in self.terminal_states:
                history_log.info('Terminal state')
                self._clear_events()
                return True

            event = self.run_state_action()
            self._receive_event(event)
            if event.wait_till:
                return True

        return False

    def _get_action_call(self):
        do = self.state_config.get('do')
        if not isinstance(do, dict):
            do = {'action': do}

        args, kwargs = do.get('args', []), do.get('kwargs', {})
        action = do['action'](self.document, 'data', self)

        if self.process.get('event_params'):
            kwargs.setdefault('event_params', self.process['event_params'])
        return action, args, kwargs

    def before_run_action(self):
        switcher.sync_with_lazy_reconnect()

    def run_state_action(self):
        try:
            action, args, kwargs = self._get_action_call()
            self.before_run_action()
            log.debug('Running %s, with args:%s, kwargs:%s', action, args, kwargs)
            return ProcessEvent.parse_from_action_result(action(*args, **kwargs))
        except Exception:
            history_log.exception('Unhandled exception')
            if self.raise_on_error:
                raise
            return self.EXCEPTION_EVENT

    def _receive_event(self, event):
        state_to = self._get_transition_state(event)
        if state_to:
            history_item = {
                'state_to': state_to,
                'event': event.name,
            }
            if event.is_external:
                history_item['external_event'] = True

            if self.process['data']:
                history_item['data'] = self.process['data']

            # doc_update не всегда сериализуется
            if event.doc_update:
                log_event = event._replace(doc_update=repr(event.doc_update))
            else:
                log_event = event

            proc_update = {
                '$set': {
                    'state': state_to,
                    'data': {},
                    'received_event': log_event._asdict(),
                    'wait_till': event.wait_till,
                    'event_params': event.params,
                    'created_at': event.created_at,
                }
            }
            self._log_history(history_item)
        else:
            msg = 'Invalid event {} for state {}'.format(event.name, self.state)
            raise ValueError(msg)

        # Убираем обработанный эвент
        if event.is_external:
            merge_updates(proc_update, {
                '$pull': {'external_events': {'uid': event.uid}},
                '$inc': {'external_events_count': -1}
            })

        self.update(proc_update, event.doc_update)

    def _clear_events(self):
        proc_update = {}
        external_events_count = len(self.process['external_events']) * -1
        merge_updates(proc_update, {
            '$pull': {
                'external_events': {
                    'uid': {'$in': [e['uid'] for e in self.process['external_events']]}
                }
            },
            '$inc': {'external_events_count': external_events_count}
        })

        self.update(proc_update)

    def _update(self, proc_update, doc_update=None, namespace=None, **kwargs):
        """
        Обновление данных в объекте процесса и в базе.
        Должна выполняться внутри блокироки document_locker, иначе UnableToUpdate

        :param proc_update: mongo-style update dict
        :param doc_update: mongoengine-style update dict (для валидации документа)
        :param namespace: расширение пути внутри неймспейса процесса
        :param kwargs: параметры в find_one_and_update
        """

        # апдейт данных документа
        doc_update = doc_update or {}
        doc_update_mongo = transform.update(self.document.__class__, **doc_update)

        # апдейт данных процесса
        namespace = self.namespace + ('.' + namespace if namespace else '')
        proc_update = get_update_with_prefix(proc_update, namespace)
        merge_updates(doc_update_mongo, proc_update)

        self.document_locker.update_document_raw(doc_update_mongo, **kwargs)

    def _log_history(self, history_item):
        try:
            history_log.info('{current_state} --> [{event}{external}] --> {state_to}'.format(
                current_state=self.state,
                state_to=history_item['state_to'],
                event=history_item['event'],
                external=' (external)' if history_item.get('external_event') else ''
            ))
        except Exception:
            history_log.exception('Ошибка логирования истории процесса')

    def _get_next_processable_event(self):
        """
        Получаем следующее событие
        """
        doc = self.document.reload()
        self.process['external_events'] = get_by_dotted_path(doc, self.namespace + '.external_events', {})
        for event_data in self.process['external_events']:
            event = ProcessEvent(**event_data)
            if self._get_transition_state(event):
                return event
        return None

    def send_external_event(self, event_name, params=None, allow_send_to_empty_process=False):
        """
        Добавляет событие в очередь
        :type event: ProcessEvent
        """
        event = ProcessEvent(event_name, params=params, is_external=True)

        proc_update = {
            '$push': {'external_events': event._asdict()},
            '$inc': {'external_events_count': 1}
        }
        doc_query = {'_id': self.document.id}

        if not (self.process and self.process.get('state')):
            if not allow_send_to_empty_process:
                proc_update['$set'] = {'state': self.EXCEPTION_STATE}

        update = get_update_with_prefix(proc_update, self.namespace)
        result = self.document_locker.collection.find_one_and_update(doc_query, update,
                                                                     return_document=ReturnDocument.AFTER)
        self.document = document_from_mongo_result(self.document.__class__, result)


class StateAction(object):
    idempotent = False

    def __init__(self, document, namespace, process):
        self.document = document
        self.namespace = namespace
        self.process = process

    def __call__(self, *args, **kwargs):
        data = self.process.get_namespace(self.namespace)
        return self.do(data, *args, **kwargs)

    def do(self, data, *args, **kwargs):
        raise NotImplementedError

    def update_data(self, update_dict):
        kwargs = {}
        if self.idempotent:
            kwargs['writeConcern'] = DEFAULT_WRITE_CONCERN
        self.process.update(update_dict, namespace=self.namespace, **kwargs)
