# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import celery

from .process import Process

_process_registry = {}


def add_process(name, scheme, model, namespace='process'):
    from .scheme import load_scheme

    if name in _process_registry:
        raise ValueError(u'Process {} is already registered'.format(name))

    _process_registry[name] = {
        'kwargs': {
            'scheme': load_scheme(scheme),
            'namespace': namespace
        },
        'model': model
    }


def get_process_model(name):
    return _process_registry[name]['model']


def get_process(name, document=None, **process_kwargs):
    document = document or get_process_model(name)()
    kwargs = _process_registry[name]['kwargs'].copy()
    kwargs.update(process_kwargs)
    return Process(document=document, **kwargs)


def clear():
    _process_registry.clear()


def get_processes_names():
    return list(_process_registry.keys())


def get_process_params(name):
    return _process_registry[name]


@celery.shared_task
def run_process(process_name, doc_id, log_context={}):
    model = _process_registry[process_name]['model']
    document = model.objects.get(id=doc_id)
    process = get_process(process_name, document)
    log_context.setdefault('process_name', process_name)
    state = process.run(log_context)
    if state == Process.NEED_RESCHEDULE_STATE:
        run_process.apply_async([process_name, doc_id, log_context])
    return process.state
