# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import os
from datetime import timedelta

from ylog.context import log_context

from common.celery.task import single_launch_task
from travel.rasp.library.python.common23.date.environment import now_utc
from common.workflow import registry

log = logging.getLogger(__name__)


class BaseReviver(object):
    def __init__(self, process_name, apply_options=None, log_context_getter=None):
        self.process_name = process_name
        self.model = registry.get_process_model(process_name)
        self.process = registry.get_process(process_name)
        self.apply_options = apply_options if apply_options else {}
        self.log_context_getter = log_context_getter

    def _ns(self, field):
        return self.process.namespace + '.' + field

    def _prepare_query(self):
        not_expired_dt = now_utc() - timedelta(seconds=self.process.document_locker.lock_alive_time)
        need_to_revive = [
            {self._ns('state'): {'$nin': self.process.terminal_states}},  # процесс не окончен
            {self._ns('external_events_count'): {'$gt': 0}},  # есть новое внешнее событие
        ]
        # TODO: remove external_events size checking after release
        if os.getenv('RASP_WORKFLOW_USE_EVENTS_SIZE', None) is not None:
            need_to_revive.append({self._ns('external_events'): {'$exists': True, '$not': {'$size': 0}}})

        query = {'$and': [
            # берём только незалоченные процессы
            {'$or': [
                {self._ns('lock_uid'): None},  # лока нет
                {self._ns('lock_modified'): {'$lt': not_expired_dt}},  # лок просрочен
            ]},
            # кроме процессов в ожидании
            {'$or': [
                {self._ns('wait_till'): None},  # не ожидал
                {self._ns('wait_till'): {'$lt': now_utc()}}  # дождался
            ]},
            # лок проверили, теперь проверяем необходимость воскрешения
            {'$or': need_to_revive}
        ]}
        return query

    def _retrieve(self, query):
        return {o['_id']: o for o in self.process.collection.find(query, {})}

    def _revive(self, objs_to_revive):
        for obj_id, obj in objs_to_revive.items():
            log_context_data = self.log_context_getter(obj) if self.log_context_getter else {}
            with log_context(**(log_context_data or {})):
                log.info('Reviving {} with document {}._id={}'.format(
                    self.process_name,
                    self.model.__name__,
                    str(obj_id)
                ))
            registry.run_process.apply_async([self.process_name, str(obj_id), log_context_data], **self.apply_options)

    def execute(self):
        self._revive(self._retrieve(self._prepare_query()))


def revive(process_name, apply_options=None, log_context_getter=None):
    """
    Воскрешает процессы workflow.Process, которые были остановлены по каким-то причинам.
    :param process_name - Имя процесса из реестра
    :param apply_options - параметры, которые будут переданы как kwargs в apply_async.
    :param log_context_getter - функция получения контекста из документа
    """
    BaseReviver(process_name, apply_options, log_context_getter).execute()


@single_launch_task()
def revive_all_processes():
    """
    Пытается воскресить все зарегистрированные в реестре процессы.
    Запускается не чаще чем раз в минуту.
    :return:
    """
    for process_name in registry.get_processes_names():
        revive(process_name, {'expiration': 60.0})
