# coding: utf8
from collections import defaultdict


nothing = object()  # value to fill default arguments


def get_by_dotted_path(obj, path, default=nothing):
    """
    С объекта obj получаем объект по пути вида 'a.b.c.d',
    причем каждый шаг в цепочке может быть либо getitem, либо getattr.

    Например, у нас может быть сочетание объекта mongoengine с обычными диктами (DynamicField) и листами:
    document.path.to.0.process['some'][2]['keys']
    """
    current = obj
    for attr in path.split('.'):
        if not attr:
            continue

        if isinstance(current, (list, tuple)):
            try:
                ind = int(attr)
            except ValueError:
                pass
            else:
                current = current[ind]
                continue
        else:
            try:
                current = current[attr]
                continue
            except (KeyError, TypeError):  # либо нет ключа, либо нет __getitem__
                pass

        try:
            current = getattr(current, attr)
        except AttributeError:
            if default is nothing:
                raise
            else:
                return default

    return current


def set_by_dotted_path(obj, path, value):
    path_parts = path.split('.')
    parent = get_by_dotted_path(obj, '.'.join(path_parts[:-1]))

    attr = path_parts[-1]
    if isinstance(parent, (list, tuple)):
        attr = int(attr)

    try:
        parent[attr] = value
    except KeyError:
        return setattr(parent, attr, value)


def get_update_with_prefix(update_dict, prefix):
    """
    {'$set': {'a': 1}} -> {'$set': {'prefix.a': 1}}
    """

    new_update_dict = defaultdict(dict)
    for oper, data in update_dict.items():
        for field, value in data.items():
            full_key = prefix + '.' + field
            new_update_dict[oper][full_key] = value

    return new_update_dict


def merge_updates(upd_to, upd_from):
    """
    Мержим два mongo update dict:

    {'$set': {'a': 1}} + {'$set': {'b': 2}} -> {'$set': {'a': 1, 'b': 2}}
    """
    for oper, data in upd_from.items():
        current = upd_to.get(oper)
        if current:
            for k, v in data.items():
                current[k] = v
        else:
            upd_to[oper] = data


def document_from_mongo_result(model, mongo_result):
    return model._from_son(mongo_result)
    # _from_son нормально обрабатывает случай, когда в монге есть лишние поля
    # но при этом не часть публичного АПИ, что плохо
    # Альтернатива без обработки лишний полей
    # result = mongo_result.copy()
    # result['id'] = result.pop('_id')
    # return model(**result)
