# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import os.path
from optparse import make_option

from django.conf import settings
from django.core.management.base import BaseCommand
from django.utils.encoding import force_unicode, smart_str

from common.utils.tanker import Tanker
from common.xgettext.common import get_common_keyset_path
from common.xgettext.management.commands.base import parse_keysets_args


class Command(BaseCommand):
    help = """https://wiki.yandex-team.ru/Raspisanija/TankerTechnical"""
    requires_model_validation = False

    def add_arguments(self, parser):
        parser.add_argument('--dry-run', action='store_true', default=False,
                    help=u'Makes no changes/modifications')
        parser.add_argument('--tanker', '-t', action='store', default='stable',
                    help=u'Tanker instance')
        parser.add_argument('--project', '-p', action='store', default='rasp',
                    help=u'Tanker project')
        parser.add_argument('--branch', '-b', action='store', default=None,
                    help=u'Tanker project branch')
        parser.add_argument('keysets', action='store', nargs='+')

    def handle(self, *args, **options):
        selected = parse_keysets_args(options['keysets'], {
            'django': settings.TANKER_KEYSETS.keys(),
            'xgettext': settings.XGETTEXT_KEYSETS.keys(),
            'common': None,
            'bem': None,
            'days_and_months': None
        })

        download(selected, options)


def download(selected, options):
    print(smart_str(u'Tanker: %s' % force_unicode(options['tanker'])))

    tanker = Tanker(tanker=options['tanker'], project=options['project'], branch=options['branch'])

    dry_run = options['dry_run']

    download_django(tanker, selected['django'], dry_run)

    if selected['bem']:
        download_bem(tanker, dry_run)

    if selected['common']:
        download_common(tanker, dry_run)

    if selected['days_and_months']:
        download_days_and_months(tanker, dry_run)

    download_xgettext(tanker, selected['xgettext'], dry_run)


def download_django(tanker, keysets, dry_run):
    for keyset in keysets:
        localedir, languages = settings.TANKER_KEYSETS[keyset]

        for keyset, (localedir, languages) in settings.TANKER_KEYSETS.items():
            pos = {}

            for lang in languages:
                path = os.path.join(settings.PROJECT_PATH, localedir, lang, 'LC_MESSAGES', 'django.po')

                pos[lang] = path

            tanker.download_po(keyset, pos, dry_run)


def download_bem(tanker, dry_run):
    # BEM
    for blocks_path, prefix, blocks in settings.XGETTEXT_BEM_BLOCKS:
        for block in blocks:
            for lang in settings.FRONTEND_LANGUAGES:
                path_elems = [settings.PROJECT_PATH]

                path_elems.extend(blocks_path)

                path_elems.extend([block, '%s.i18n' % block, '%s.js' % lang])

                path = os.path.join(*path_elems)

                dirname = os.path.dirname(path)

                if not os.path.exists(dirname):
                    os.mkdir(dirname)

                tanker.download_bem(block, lang, path, prefix=prefix)


def download_common(tanker, dry_run):
    tanker.download_tjson('datetemplates', settings.FRONTEND_LANGUAGES,
                          get_common_keyset_path('datetemplates'), dry_run,
                          strip_forms=True)

    tanker.download_tjson('schedule_autotexts', settings.FRONTEND_LANGUAGES,
                          get_common_keyset_path('schedule_autotexts'), dry_run,
                          strip_forms=True)


def download_days_and_months(tanker, dry_run):
    tanker.download_lang_keyset_key_json(
        'days',
        settings.FRONTEND_LANGUAGES,
        get_common_keyset_path('days'),
        dry_run
    )

    tanker.download_lang_keyset_key_json(
        'months',
        settings.FRONTEND_LANGUAGES,
        get_common_keyset_path('months'),
        dry_run
    )


def download_xgettext(tanker, keysets, dry_run):
    for keyset in keysets:
        attrs = settings.XGETTEXT_KEYSETS[keyset]

        keyset_name = 'xgettext:' + keyset
        keyset_path = os.path.join(settings.PROJECT_PATH, attrs['filename'])

        tanker.download_tjson(keyset_name, settings.XGETTEXT_LANGUAGES, keyset_path, dry_run)
