# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import ast
import os.path
import sys
from optparse import make_option
from textwrap import dedent

from django.conf import settings
from django.core.management.base import BaseCommand
from django.utils.encoding import force_text, smart_bytes

from common.utils.tanker import Tanker, tanker_json_dumps
from common.xgettext.i18n import xgettexttree, form_key
from common.xgettext.management.commands.base import parse_keysets_args


class Command(BaseCommand):
    help = """https://wiki.yandex-team.ru/Raspisanija/TankerTechnical"""

    def add_arguments(self, parser):
        parser.add_argument('--dry-run', action='store_true', default=False,
                    help=u'Makes no changes/modifications')
        parser.add_argument('--tanker', '-t', action='store', default='stable',
                    help=u'Tanker instance')
        parser.add_argument('--token', '-u', action='store', default=None,
                    help=u'Project token')
        # Описание режимов загрузки см.
        # http://doc.yandex-team.ru/Tanker/api-reference/concepts/operations-translations.xml#trans-add
        parser.add_argument('--mode', '-m', action='store', default='onlyadd',
                    help=u'Upload mode: create|replace|onlyadd|merge')
        parser.add_argument('--project', '-p', action='store', default='rasp',
                    help=u'Tanker project')
        parser.add_argument('--branch', '-b', action='store', default=None,
                    help=u'Tanker project branch')
        parser.add_argument('keysets', action='store', nargs='+')
    
    requires_model_validation = False

    def handle(self, *args, **options):
        selected = parse_keysets_args(options['keysets'], {
            'django': settings.TANKER_KEYSETS.keys(),
            'xgettext': settings.XGETTEXT_KEYSETS.keys(),
        })

        upload(selected, options)


def upload(selected, options):
    print(smart_bytes(u'Tanker: %s' % force_text(options['tanker'])))

    dry_run = options['dry_run']

    tanker = Tanker(tanker=options['tanker'], token=options['token'],
                    dry_run=dry_run, mode=options['mode'],
                    project=options['project'], branch=options['branch'])

    upload_django(tanker, selected['django'], dry_run)
    upload_xgettext(tanker, selected['xgettext'], dry_run)


def upload_django(tanker, keysets, dry_run):
    for keyset in keysets:
        localedir, languages = settings.TANKER_KEYSETS[keyset]

        pos = {}

        for lang in languages:
            path = os.path.join(settings.PROJECT_PATH, localedir, lang, 'LC_MESSAGES', 'django.po')

            if os.path.exists(path):
                pos[lang] = path

        print("Keyset", keyset)

        if dry_run:
            print(tanker_json_dumps(pos).encode('utf-8'))

        tanker.upload_po(keyset, pos)


def upload_xgettext(tanker, keysets, dry_run):
    for keyset in keysets:
        attrs = settings.XGETTEXT_KEYSETS[keyset]

        keys = {}

        for basepath in attrs['dirs']:
            realpath = os.path.join(settings.PROJECT_PATH, basepath)

            for dirpath, _dirnames, filenames in os.walk(realpath):
                for filename in filenames:
                    _root, ext = os.path.splitext(filename)

                    if ext != '.py':
                        continue

                    source = os.path.join(dirpath, filename)

                    path = os.path.relpath(source, settings.PROJECT_PATH)

                    if os.path.exists(source):
                        extract_from_file(keys, source, path)

        keyset_name = 'xgettext:' + keyset

        print("Keyset", keyset_name)

        if dry_run:
            print(tanker_json_dumps(keys).encode('utf-8'))

        tanker.upload(keyset_name, keys)


class FormException(Exception):
    def __init__(self, msg, text):
        self.msg = msg
        self.text = text


class Context(object):
    def __init__(self):
        self.references = []

    def add(self, filename, lineno, comments):
        self.references.append((filename, lineno, comments))

    def __json__(self):
        return "\n\n".join(
            "\n".join(["%s:%d" % (filename, lineno)] + comments)
            for filename, lineno, comments in self.references
        )


class SingleForm(object):
    def __init__(self, form):
        self.form = form
        self.context = Context()

    def __json__(self):
        return {
            "info": {
                "is_plural": False,
                "context": self.context
            },
            "translations": {
                settings.BASE_LANGUAGE: {
                    "status": "approved",
                    "form": self.form
                }
            }
        }


class PluralForms(object):
    def __init__(self, forms):
        self.forms = forms
        self.context = Context()

    def __json__(self):
        translation = {
            "status": "approved",
        }

        for i, form in enumerate(self.forms):
            translation["form%d" % (i + 1)] = form

        return {
            "info": {
                "is_plural": True,
                "context": self.context
            },
            "translations": {
                settings.BASE_LANGUAGE: translation
            }
        }


def extract_from_file(keys, path, filename):
    with open(path) as f:
        source = f.read()

    lines = source.decode('utf-8').split('\n')

    root = ast.parse(source, filename)

    def prepare_form(form_arg, xgettext=False):
        if not isinstance(form_arg, ast.Str):
            raise FormException("Invalid %s argument at %s:%d" % (func, filename, lineno), lines[lineno - 1])

        try:
            form_arg_s = force_text(form_arg.s)
        except UnicodeDecodeError:
            raise FormException("Could not convert argument to unicode at %s:%d" % (filename, lineno), repr(form_arg.s))

        form = dedent(form_arg_s).strip()

        if xgettext:
            try:
                xgettexttree(form)
            except Exception as e:
                raise FormException("Invalid form xml %s at %s:%d:" % (e, filename, lineno), form)

        return form

    for node in ast.walk(root):
        if isinstance(node, ast.Call):
            if isinstance(node.func, ast.Name):
                func = node.func.id

                if func in ['gettext', 'mark_gettext', 'ngettext', 'xgettext', 'xngettext', 'tgettext', 'tngettext']:
                    lineno = node.lineno

                    line = lines[lineno - 1]

                    from pygments.lexers import PythonLexer
                    from pygments.token import Token

                    comments = []

                    for tokentype, value in PythonLexer().get_tokens(line):
                        if tokentype is Token.Comment:
                            comments = [value]
                            break

                    if not comments:
                        prev_line = lineno - 1

                        while prev_line > 0:
                            line = lines[prev_line - 1]

                            prev_line -= 1

                            stripped = line.strip()

                            if not stripped:
                                continue

                            if stripped.startswith('#'):
                                comments.insert(0, stripped)
                            else:
                                break

                    try:
                        key = None

                        for keyword in node.keywords:
                            if keyword.arg == 'key':
                                if not isinstance(keyword.value, ast.Str):
                                    raise FormException("Invalid %s key at %s:%d" %
                                                        (func, filename, lineno), lines[lineno - 1])

                                key = keyword.value.s

                        if func in ['gettext', 'mark_gettext', 'xgettext', 'tgettext']:
                            form_arg = node.args[0]

                            form = prepare_form(form_arg, func == 'xgettext')

                            if not key:
                                key = form_key(form)

                            if key not in keys:
                                keys[key] = SingleForm(form)

                            assert keys[key].form == form

                        else:
                            forms = [prepare_form(arg, func in ['xngettext', 'tngettext']) for arg in node.args[1:]]

                            if not key:
                                key = form_key(forms[0])

                            if key not in keys:
                                keys[key] = PluralForms(forms)

                            assert keys[key].forms == forms

                        keys[key].context.add(filename, lineno, comments)

                    except FormException as e:
                        print(e.msg, file=sys.stderr)
                        print(e.text, file=sys.stderr)
