# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from typing import List  # noqa

from enum import Enum
from yandex.cloud.mdb.mongodb.v1.cluster_service_pb2 import (
    ListClusterHostsRequest as MongoListClusterHostsRequest
)
from yandex.cloud.mdb.mongodb.v1.cluster_service_pb2_grpc import (
    ClusterServiceStub as MongoClusterServiceStub
)
from yandex.cloud.mdb.postgresql.v1.cluster_service_pb2 import (
    ListClusterHostsRequest as PostgresqlListClusterHostsRequest
)
from yandex.cloud.mdb.postgresql.v1.cluster_service_pb2_grpc import (
    ClusterServiceStub as PostgresqlClusterServiceStub
)

from travel.rasp.library.python.common23.data_api.dbaas.host_info import HostInfo


class UnknownDbaasBaseType(Exception):
    pass


class DbType(Enum):
    MONGODB = "mongodb"
    POSTGRESQL = "postgresql"


class DbaasClient(object):
    def __init__(self, api_url, oauth_token, root_cert_file=None, host_sorter=None):
        root_cert = open(root_cert_file).read() if root_cert_file else None

        self.oauth_token = oauth_token
        self.api_url = api_url
        self.root_cert = root_cert
        self._host_sorter = host_sorter
        self._sdk = None

    @property
    def sdk(self):
        if not self._sdk:
            import yandexcloud
            self._sdk = yandexcloud.SDK(
                token=self.oauth_token,
                endpoint=self.api_url,
                root_certificates=self.root_cert,
            )
        return self._sdk

    def _get_hosts(self, cluster_id, clusterServiceStub, listClusterHostsRequest):
        cluster = self.sdk.client(clusterServiceStub)
        response = cluster.ListHosts(listClusterHostsRequest(cluster_id=cluster_id))
        return response.hosts

    def get_hosts(self, cluster_id, db_type):
        # type: (unicode, DbType) -> List[HostInfo]

        clusterServiceStub = None
        listClusterHostsRequest = None
        if db_type == DbType.MONGODB:
            clusterServiceStub = MongoClusterServiceStub
            listClusterHostsRequest = MongoListClusterHostsRequest
        elif db_type == DbType.POSTGRESQL:
            clusterServiceStub = PostgresqlClusterServiceStub
            listClusterHostsRequest = PostgresqlListClusterHostsRequest
        else:
            raise UnknownDbaasBaseType(dict(
                cluster_id=cluster_id,
                db_type=db_type,
            ))

        hosts = self._get_hosts(cluster_id, clusterServiceStub, listClusterHostsRequest)
        host_infos = [HostInfo(host.name, host.zone_id) for host in hosts]

        if self._host_sorter is not None:
            host_infos = self._host_sorter.sort(host_infos)

        return host_infos
