# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from time import time
from django.conf import settings
from future.utils import python_2_unicode_compatible

from errorboosterclient.logbroker import LogbrokerClient
from errorboosterclient.sentry import EventConverter
from raven.conf.remote import RemoteConfig
from raven.contrib.django.client import DjangoClient
from raven.transport.base import Transport
from raven.utils.json import BetterJSONEncoder

from travel.rasp.library.python.common23.settings.configuration import Configuration
from travel.rasp.library.python.common23.settings.utils import define_setting
from travel.rasp.library.python.common23.utils.dcutils import resource_explorer


ERROR_BOOSTER_PROJECT = 'rasp'

define_setting(
    'ERROR_BOOSTER_TOPIC',
    {Configuration.PRODUCTION: 'rasp/rasp-production-error-booster'},
    default='rasp-front/rasp-testing-error-booster'
)


def sender(event):
    topic = settings.ERROR_BOOSTER_TOPIC
    dc = resource_explorer.get_current_dc()

    with LogbrokerClient(
        token=settings.LOGBROKER_OAUTH_TOKEN
    ) as logbroker:
        with logbroker.get_producer(
            topic=topic,
            source='{}_{}'.format(dc, event['service']),
            json_encoder=BetterJSONEncoder
        ) as producer:
            producer.write(event)


class RavenErrorBoosterClient(DjangoClient):
    ERROR_BOOSTER_CONVERTER = EventConverter

    def __init__(self, service_name, project_name=ERROR_BOOSTER_PROJECT, logbroker_sender=sender, *args, **kwargs):
        self.service_name = service_name
        self.project_name = project_name
        self.logbroker_sender = logbroker_sender

        super(RavenErrorBoosterClient, self).__init__(*args, **kwargs)

    def set_dsn(self, dsn=None, transport=None):
        if dsn not in self._transport_cache:
            result = RavenErrorBoosterRemoteConfig(
                transport=transport,
                options={'logbroker_sender': self.logbroker_sender},
                project=self.project_name
            )
            self._transport_cache[dsn] = result
            self.remote = result
        else:
            self.remote = self._transport_cache[dsn]
        self.logger.debug("Configuring Raven for host: %s", self.remote)

    def decode(self, data):
        if isinstance(data, dict):
            return data

        return super(RavenErrorBoosterClient, self).decode(data)

    def get_full_service_name(self):
        return '{}_{}_{}'.format(self.project_name, self.service_name, self.environment)

    def get_str_level(self, int_level):
        return logging.getLevelName(int_level).lower()

    def error_booster_convert(self, data):
        data['timestamp_'] = int(round(time() * 1000))  # бустер ожидает миллисекунды
        converted = self.ERROR_BOOSTER_CONVERTER.convert(data)
        converted['project'] = self.project_name
        converted['service'] = self.service_name
        converted['level'] = self.get_str_level(converted['level'])
        request_data = data.get('request')
        converted['additional']['request'] = request_data
        if request_data and 'url' in request_data:
            converted['url'] = request_data['url']
        return converted

    def send(self, auth_header=None, **data):
        converted_data = self.error_booster_convert(data)

        return self.send_remote(
            url=self.remote.store_endpoint,
            data=converted_data,
        )


@python_2_unicode_compatible
class RavenErrorBoosterRemoteConfig(RemoteConfig):
    def __init__(self, *args, **kwargs):
        super(RavenErrorBoosterRemoteConfig, self).__init__(*args, **kwargs)
        self.store_endpoint = 'error booster project: {}'.format(self.project)

    def is_active(self):
        return bool(self.project)

    def __str__(self):
        return self.store_endpoint


class RavenErrorBoosterTransport(Transport):
    def __init__(self, logbroker_sender):
        self.logbroker_sender = logbroker_sender

    def send(self, url, data, headers):
        self.logbroker_sender(data)
