# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from builtins import str
import logging

from django.utils.encoding import force_str

from travel.rasp.library.python.common23.data_api.juggler.instance import juggler_client
from travel.rasp.library.python.common23.data_api.juggler.env_params import get_env_params

log = logging.getLogger(__name__)

OK, WARN, CRIT = 'OK', 'WARN', 'CRIT'


class CheckError(Exception):
    def __init__(self, status, description):
        self.status = status
        self.description = description


class Event(object):
    def __init__(self, service, status=OK, description=OK, tags=None, host=None):
        self.service = service
        self.status = status
        self.description = description
        self.tags = ['scripts']
        self.host = host
        if tags:
            self.tags.extend(tags)

    def __str__(self):
        return force_str('{}: {}; {}; {}; {}'.format(self.service, self.status, self.tags, self.description, self.host))


def run_monitoring(check_func):
    try:
        result = check_func()
    except CheckError as ex:
        return ex.status, ex.description
    except Exception as ex:
        log.exception('%s error:', check_func)
        return CRIT, repr(ex)
    else:
        ok_msg = OK if result is None else result
        return OK, str(ok_msg)


def get_events(service_checks):
    events = []
    for service, check, tags in service_checks:
        status, description = run_monitoring(check)
        event = Event(service=service, status=status, description=description, tags=tags)
        events.append(event)
        log.info('Event: %s', event)
    return events


def send_events(events, source=None):
    current_env = get_env_params()

    juggler_client.send([{
            'description': event.description,
            'host': event.host or current_env.get('host'),
            'instance': '',
            'service': event.service,
            'status': event.status,
            'tags': event.tags
        } for event in events], source=source)
