# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import calendar
import logging
import pytz
import re
import six
import time
import warnings
from collections import namedtuple
from datetime import date, datetime, timedelta
from functools import total_ordering
from itertools import groupby
from pytils.dt import ru_strftime
from time import mktime

from django.conf import settings
from django.utils import translation
from django.utils.encoding import force_text
from django.utils.lru_cache import lru_cache

from travel.rasp.library.python.common23.date import environment
from travel.rasp.library.python.common23.date.date_const import MSK_TZ, MSK_TIMEZONE
from travel.rasp.library.python.common23.utils.text import NBSP
from travel.rasp.library.python.common23.utils.warnings import RaspDeprecationWarning
from travel.rasp.library.python.common23.xgettext.common import (
    xgettext_weekday, xgettext_weekday_short, xgettext_month, xgettext_month_genitive, xgettext_month_short
)

log = logging.getLogger(__name__)

# заполняем числами - сколько дней в каждом месяце этого года
days_in_months = []


def get_days_in_months(fresh=False):
    global days_in_months
    if days_in_months and not fresh:
        return days_in_months

    # Чтобы в високосные года брался более актуальный февраль
    year = (datetime.today() + timedelta(30 * 8)).year

    days_in_months = []
    for i in range(1, 12):
        days_in_months.append((datetime(year, i + 1, 1) - timedelta(1)).day)
    days_in_months.append(31)
    return days_in_months


def get_short_yeardays(yeardays, mask=get_days_in_months(True)):
    """ Возвращает год в виде строки.
        В каждом месяце столько дней сколько должно быть, а не 31.
    """
    res = ''
    for i in range(len(mask)):
        days = get_days_in_months()[i]
        res += yeardays[i * 31:i * 31 + days]
    return res


def get_full_yeardays(yeardays):
    """ Возвращает год в виде строки.
        В каждом месяце ровно 31 день.
    """
    res = ''
    for i in range(len(get_days_in_months())):
        days = get_days_in_months()[i]
        res += yeardays[:days] + '0' * (31 - days)
        yeardays = yeardays[days:]
    return res


def get_moved_year_days(year_days, sds):
    """ Возвращает сдвинутые на sds дней year_days
    """
    return year_days[-sds:] + year_days[:-sds]


def get_first_month_days(months=12, today=None):
    """ Формирует список дат - первых дней месяцев, начиная от предыдущего перед текущим
        Вход: количество месяцев
        Выход: список первых дней месяцев """
    if not today:
        today = environment.today()
    start = date(today.month == 1 and today.year - 1 or today.year,
                 today.month == 1 and 12 or today.month - 1,
                 1)

    def get_year(m):
        return start.month + m > 12 and start.year + 1 or start.year

    def get_month(m):
        return start.month + m == 12 and 12 or (start.month + m) % 12

    return (date(get_year(m), get_month(m), 1) for m in range(0, months))


def fromSQLdatetime(datetimeStr):
    """Возвращает дату-время из SQL-формата
    Вход: дата-время в формате yyyy-mm-dd hh:mm:ss
    Выход: объект datetime"""
    return datetime(*time.strptime(datetimeStr, "%Y-%m-%d %H:%M:%S")[:6])


def fromSQLDate(dateStr):
    """Возвращает дату из SQL-формата
    Вход: дата в формате yyyy-mm-dd
    Выход: объект date"""
    return datetime(*time.strptime(dateStr, "%Y-%m-%d")[:3])


def parse_date(dt):
    if dt is None:
        return None

    return date(*(int(e) for e in dt.split('-')))


def timedelta2minutes(delta):
    return delta.total_seconds() / 60


def timedelta2hours(delta):
    return delta.total_seconds() / 3600


def timedelta_to_str_hours_and_minutes(td):
    """
    :param td: datetime.timdedelta
    :return: unicode: HH:MM

    Ex: timedelta(hours=1, minutes=42) => '01:42'
    """
    hours, min_sec = divmod(td.total_seconds(), 3600)
    minutes, _ = divmod(min_sec, 60)

    return u'{:02d}:{:02d}'.format(int(hours), int(minutes))


def get_pytz(time_zone):
    try:
        return pytz.timezone(str(time_zone))
    except pytz.UnknownTimeZoneError:
        return MSK_TZ


@lru_cache()
def get_timezone_from_point(point):
    time_zone = None

    if isinstance(point, dict):
        time_zone = point['time_zone'] or MSK_TIMEZONE
    elif isinstance(point, six.string_types):
        return point
    else:
        if hasattr(point, 'settlement') and point.settlement:
            time_zone = point.settlement.time_zone

        if time_zone is None:
            time_zone = point.time_zone

        if time_zone is None and hasattr(point, 'region'):
            time_zone = point.region and point.region.time_zone

    if time_zone is None:
        time_zone = MSK_TIMEZONE

    return time_zone


def get_msk_time_zone(point, msk_datetime=None, local_datetime=None,
                      utc_datetime=None):
    """ Возвращает смещение географического объекта в минутах относительно москвы
    с учетом перехода его на зимнее время,
    принимется либо местное время либо Московское

    Объект должен быть словарем содержищим ключи 'time_zone' и 'switch_time',
    или объектом содержищим такие аттрибуты,
    или объектом django моделей Settlement, Station или Region

    Если не указано время, то берем текущее.

    """

    if msk_datetime is None and local_datetime is None:
        msk_datetime = datetime.now()

    time_zone = get_timezone_from_point(point)

    loc_tz = get_pytz(time_zone)

    utc_tz = pytz.utc

    if msk_datetime is not None:
        msk_datetime = MSK_TZ.localize(msk_datetime)
        local_datetime = msk_datetime.astimezone(loc_tz)
    elif local_datetime is not None:
        local_datetime = loc_tz.localize(local_datetime)
        msk_datetime = local_datetime.astimezone(MSK_TZ)
    elif utc_datetime is not None:
        utc_datetime = utc_tz.localize(utc_datetime)
        msk_datetime = utc_datetime.astimezone(MSK_TZ)
        local_datetime = utc_datetime.astimezone(loc_tz)

    return timedelta2minutes(local_datetime.replace(tzinfo=None) - msk_datetime.replace(tzinfo=None))


def get_local_time(point, msk_datetime=None, utc_datetime=None):
    """ Конвертирует Московское время в местное c учетом того,
    переходит географический объект на зимнее время или нет.

    Объект должен быть словарем содержищим ключи 'time_zone' и 'switch_time',
    или объектом содержищим такие аттрибуты,
    или объектом django моделей Settlement, Station или Region
    """

    time_zone = get_timezone_from_point(point)

    if msk_datetime:
        loc_tz = get_pytz(time_zone)
        msk_datetime = MSK_TZ.localize(msk_datetime)
        local_datetime = msk_datetime.astimezone(loc_tz)
    elif utc_datetime:
        utc_tz = pytz.utc
        loc_tz = get_pytz(time_zone)
        utc_datetime = utc_tz.localize(utc_datetime)
        local_datetime = utc_datetime.astimezone(loc_tz)
    else:
        raise ValueError("No datetime")

    return local_datetime


class Localizer(object):
    def __init__(self, point):
        self.loc_tz = get_pytz(get_timezone_from_point(point))

    def localize(self, msk_datetime):
        if self.loc_tz == MSK_TZ:
            return MSK_TZ.localize(msk_datetime)
        else:
            return MSK_TZ.localize(msk_datetime).astimezone(self.loc_tz)


def get_msk_time(point, local_datetime):
    """ Конвертирует местное время в Московкое c учетом того,
    переходит географический объект на зимнее время или нет.

    Объект должен быть словарем содержищим ключи 'time_zone' и 'switch_time',
    или объектом содержищим такие аттрибуты,
    или объектом django моделей Settlement, Station или Region
    """

    time_zone = get_timezone_from_point(point)

    loc_tz = get_pytz(time_zone)

    local_datetime = loc_tz.localize(local_datetime)
    return local_datetime.astimezone(MSK_TZ)


def get_utc_time(point, local_datetime):
    if isinstance(point, dict):
        time_zone = point['time_zone'] or MSK_TIMEZONE
    else:
        time_zone = point.time_zone or MSK_TIMEZONE
        # На то случай если это None

    loc_tz = get_pytz(time_zone)
    utc_tz = pytz.utc

    local_datetime = loc_tz.localize(local_datetime)
    return local_datetime.astimezone(utc_tz)


def daterange(begin, end, delta=timedelta(1), include_end=False):
    """Form a range of dates and iterate over them.

    Arguments:
    begin -- a date (or datetime) object; the beginning of the range.
    end   -- a date (or datetime) object; the end of the range.
    delta -- (optional) a timedelta object; how much to step each iteration.
             Default step is 1 day.

    Usage:

    """
    if not isinstance(delta, timedelta):
        delta = timedelta(delta)

    zero_delta = timedelta(0)

    if begin == end and abs(delta) == timedelta(1) and include_end:
        yield begin
        return

    if begin < end:
        if delta <= zero_delta:
            return
        if include_end:
            test = end.__ge__
        else:
            test = end.__gt__
    else:
        if delta >= zero_delta:
            return
        if include_end:
            test = end.__le__
        else:
            test = end.__lt__

    while test(begin):
        yield begin
        begin += delta


def fix_date(d, today):
    """Выставляет дате год с учетом наших ограничений"""
    if d and not((d.month, d.day) == (2, 29)):
        left_border = today - timedelta(settings.DAYS_TO_PAST)

        d = d.replace(year=today.year)

        if d < left_border:
            d = d.replace(year=today.year + 1)

    return d


@six.python_2_unicode_compatible
class DaysText(namedtuple('DaysText', 'raw has_more')):
    def __str__(self):
        if self.has_more:
            return self.raw + u', …'

        return self.raw


def get_schedule_change_period(today):
    warnings.warn('Use this TrainSchedulePlan.get_schedule_end_period ## 2015-04-14',
                  RaspDeprecationWarning, stacklevel=2)

    # RASP-4269

    # Последнее воскресенье мая, но не ранее, чем сегодня
    may_end = today.replace(month=5, day=31)

    last_sun = may_end - timedelta((may_end.weekday() + 1) % 7)

    # Берем следующее последнее воскресенье мая
    if last_sun < today:
        may_end = may_end.replace(year=today.year + 1)
        last_sun = may_end - timedelta((may_end.weekday() + 1) % 7)

    april_first = last_sun.replace(month=4, day=1)

    return april_first, last_sun


def group_days(days, lang='ru', days_month_separator=NBSP):
    """
    :param days: [(day, month)]
    :type days: [(int, int)]
    :param lang: язык
    :return: Форматированные дни хождения
    """
    if not days:
        return u''

    if not isinstance(days[0], tuple):
        days = [tuple(int(p) for p in _d.split(u'.')) for _d in days.split(u',')]

    grouped = groupby(days, key=lambda d: d[1])

    daysparts = []

    group_days_format = u'{{}}{separator}{{}}'.format(separator=days_month_separator)

    for month, days in grouped:
        days = u', '.join(str(d[0]) for d in days)
        daysparts.append(group_days_format.format(days, xgettext_month_genitive(month, lang).lower()))

    return u', '.join(daysparts)


def to_msk(dt):
    return dt.astimezone(MSK_TZ).replace(tzinfo=None)


@total_ordering
class FuzzyDateTime(object):
    STEP = 5
    HALF_STEP = float(STEP) / 2

    timetuple = True

    def __init__(self, dt, round_dt=None):
        self.dt = dt
        self.round_dt = round_dt or dt

    @classmethod
    def fuzz(cls, dt):
        # Разница с округленным значением
        round_diff = (dt.minute + cls.HALF_STEP) % cls.STEP - cls.HALF_STEP

        round_dt = dt - timedelta(
            microseconds=dt.microsecond,
            seconds=dt.second,
            minutes=round_diff
        )

        return cls(dt, round_dt)

    def __lt__(self, other):
        if isinstance(other, self.__class__):
            return self.round_dt < other.round_dt

        if isinstance(other, datetime):
            return self.round_dt < other

        return NotImplemented

    def __eq__(self, other):
        if isinstance(other, self.__class__):
            return self.round_dt == other.round_dt

        if isinstance(other, datetime):
            return self.round_dt == other

        return NotImplemented

    def astimezone(self, *args, **kwargs):
        return self.__class__(self.round_dt.astimezone(*args, **kwargs))

    def replace(self, *args, **kwargs):
        return self.__class__(self.round_dt.replace(*args, **kwargs))

    def __add__(self, other):
        return self.round_dt + other

    def __sub__(self, other):
        if isinstance(other, self.__class__):
            return self.round_dt - other.round_dt

        return self.round_dt - other

    def __rsub__(self, other):
        if isinstance(other, self.__class__):
            return other.round_dt - self.round_dt

        return other - self.round_dt

    def __str__(self):
        return self.strftime('%Y-%m-%d')

    def __getattr__(self, item):
        return getattr(self.round_dt, item)


def astimezone(dt, city):
    # tz передается как город

    if not city:
        # местное время
        return dt

    return dt.astimezone(get_pytz(city.time_zone))


def _uni_strftime(fmt, dt, inflected=False, lang=None):
    fmt = re.sub(r'(?<![\.\d\-])%d(?![\.\d\-])', force_text(dt.day), fmt, flags=re.M + re.U)
    if u'%b' in fmt or u'%B' in fmt:
        fmt = fmt.replace(u'%d', force_text(dt.day))

    weekday = dt.weekday() + 1  # пн = 1,..., вс = 7

    lang = lang or translation.get_language()

    month_func = xgettext_month_genitive if inflected else xgettext_month

    fmt = fmt.replace(u'%a', xgettext_weekday_short(weekday, lang))
    fmt = fmt.replace(u'%A', xgettext_weekday(weekday, lang))
    fmt = fmt.replace(u'%b', xgettext_month_short(dt.month, lang))

    if '%B' in fmt:
        # Для этих языков по умолчанию писать месяца с маленькой буквы
        if lang in ('ru', 'uk'):
            month = month_func(dt.month, lang).lower()
        else:
            month = month_func(dt.month, lang)

        fmt = fmt.replace(u'%B', month)

    return dt.strftime(fmt.encode('utf-8')).decode('utf-8')


def uni_strftime(fmt, dt, lang=None):
    lang = lang or translation.get_language()

    fmt = force_text(fmt)

    if isinstance(dt, FuzzyDateTime):
        dt = dt.round_dt

    inflected = u'%d %B' in fmt or (u'%d' + NBSP + u'%B') in fmt

    if lang == 'ru':
        return ru_strftime(fmt, dt, inflected=inflected)

    else:
        return _uni_strftime(fmt, dt, inflected=inflected, lang=lang)


class DateTimeFormatter(object):
    def __init__(self, dt, lang=None):
        self.dt = dt
        self.lang = lang

    def L(self, format=u'%d %B'):
        return uni_strftime(format, self.dt, self.lang)

    def __format__(self, format_):
        return self.L(format_)


def human_date_without_year(dt):
    return uni_strftime(u'%d %B', dt)


def human_date(dt):
    if environment.now_aware().year == dt.year:
        return human_date_without_year(dt)

    return uni_strftime(u'%d %B %Y', dt)


def human_date_with_time(dt):
    return uni_strftime(u'%d %B %H:%M', dt)


def get_plural_form(n, forms):
    return u'%d %s' % (
        n,
        forms[
            0 if (n % 10 == 1 and n % 100 != 11)
            else (1 if (n % 10 >= 2 and n % 10 <= 4 and (n % 100 < 10 or n % 100 >= 20)) else 2)]
    )


def human_duration(duration):
    if not duration or duration < timedelta():
        return None

    minutes = (duration.seconds + 1) // 60
    hours = 0
    days = duration.days

    if minutes > 59:
        hours = minutes // 60
        minutes %= 60

    # Если есть дни, минуты округляем до часов
    if days:
        hours += (minutes + 60 - 1) // 60
        minutes = 0

        # Если получилось 24 часа, то прибавляем день и убираем часы
        if hours == 24:
            days += 1
            hours = 0

    blocks = []

    if days:
        # Продолжительность действия: 1 день. Может являться частью фразы, например, 1 день 2 часа
        blocks.append(get_plural_form(days, [u'день', u'дня', u'дней']))

    if hours:
        # Продолжительность действия: 1 час. Может являться частью фразы, например, 1 день 2 часа
        blocks.append(get_plural_form(hours, [u'ч', u'ч', u'ч']))

    if minutes and len(blocks) < 2:
        # Продолжительность действия: 1 минута. Может являться частью фразы, например, 2 часа 3 минуты
        blocks.append(get_plural_form(minutes, [u'мин', u'мин', u'мин']))

    return u' '.join(b.replace(' ', NBSP) for b in blocks)


def get_time_zone_choices():
    # Немного американской тематики
    summer_day = datetime(date.today().year, 7, 4)
    # Новый год, новый год
    winter_day = datetime(date.today().year, 1, 1)

    if datetime.today() > summer_day:
        winter_day = datetime(date.today().year, 12, 1)

    msk_datetime_s = MSK_TZ.localize(summer_day)
    msk_datetime_w = MSK_TZ.localize(winter_day)

    def get_diff_hours(tz):
        loc_tz = pytz.timezone(tz)
        loc_datetime = msk_datetime_s.astimezone(loc_tz)
        diff = loc_datetime.replace(tzinfo=None) - msk_datetime_s.replace(tzinfo=None)
        summer_hours = diff.days * 24 + diff.seconds / 3600.0

        loc_datetime = msk_datetime_w.astimezone(loc_tz)
        diff = loc_datetime.replace(tzinfo=None) - msk_datetime_w.replace(tzinfo=None)
        winter_hours = diff.days * 24 + diff.seconds / 3600.0

        return u" %+.2f от Москвы летом, %+.2f от Москвы зимой" % (summer_hours, winter_hours)

    choices = [(force_text(tz), force_text(tz) + get_diff_hours(tz))
               for tz in sorted(pytz.all_timezones)]
    return choices


def to_unix_timestamp(dt):
    """
    Преобразует дату со временем в unix timestamp
    Но таймстемп будет смещен относительно временной зоны на машине
    """

    return int(mktime(dt.timetuple()))


def naive_to_timestamp(dt, epoch=datetime(1970, 1, 1)):
    td = dt - epoch
    return int(td.total_seconds())


def smart_localize(dt, tz):
    """
    Если время может быть как зимним так и летним, берем наибольшее по UTC
    """
    try:
        return tz.localize(dt, is_dst=None)
    except pytz.InvalidTimeError:
        pass

    return max(
        tz.normalize(tz.localize(dt, is_dst=False)),
        tz.normalize(tz.localize(dt, is_dst=True))
    )


def get_last_day_of_month_index(year, month):
    return calendar.monthrange(year, month)[1]
