# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import warnings
from datetime import date, datetime, timedelta
from itertools import dropwhile, takewhile

from django.conf import settings
from django.utils import translation
from travel.rasp.library.python.common23.date.date import daterange, DaysText, group_days
from travel.rasp.library.python.common23.utils.warnings import RaspDeprecationWarning

log = logging.getLogger(__name__)


DAYS_TEXT_LIMIT_MIN = 7
DAYS_TEXT_LIMIT_MAX = 20


class RunMask(object):
    """ Маска дней хождения"""

    MASK_LENGTH = 12 * 31
    EMPTY_YEAR_DAYS = '0' * MASK_LENGTH
    ALL_YEAR_DAYS = '1' * MASK_LENGTH

    cached = {}  # Маски дат для разных дней

    _extrapolation_forbidden = False

    def _get_extrapolation_forbidden(self):
        # _extrapolation_forbidden не связано с маской напрямую и хранить его тут не стоит
        warnings.warn(u'Bad Manner: extrapolation_forbidden ## 2016-11-17', RaspDeprecationWarning, stacklevel=2)
        return self._extrapolation_forbidden

    def _set_extrapolation_forbidden(self, value):
        # _extrapolation_forbidden не связано с маской напрямую и хранить его тут не стоит
        warnings.warn(u'Bad Manner: extrapolation_forbidden ## 2016-11-17', RaspDeprecationWarning, stacklevel=2)
        self._is_extrapolatable = value

    extrapolation_forbidden = property(_get_extrapolation_forbidden, _set_extrapolation_forbidden)

    def __init__(self, mask=None, today=None, days=None, strict=False):
        if hasattr(mask, '__int__'):
            self.mask = int(mask)
        elif mask is None or mask == '':
            self.mask = 0
        else:
            if len(mask) == self.MASK_LENGTH:
                try:
                    self.mask = int(mask, 2)
                except (TypeError, ValueError):
                    if strict:
                        raise ValueError("Str mask must have %s chars instead of %s" % (self.MASK_LENGTH, len(mask)))
                    else:
                        log.error("Bad runmask %r" % mask)
                        self.mask = 0
            else:
                if strict:
                    raise ValueError("Str mask must have %s chars instead of %s" % (self.MASK_LENGTH, len(mask)))
                else:
                    log.error("Bad runmask %r" % mask)
                    self.mask = 0

        if days:
            for day in days:
                self[day] = 1

        self.set_today(today)

    def get_today(self):
        return self._today

    def set_today(self, today):
        if isinstance(today, datetime):
            today = today.date()

        self._today = today

        # для разных переборов
        if self._today:

            if self._today in self.cached:
                self.all_days, self.date_masks = self.cached[self._today]
            else:
                start_day = self._today - timedelta(settings.DAYS_TO_PAST)

                try:
                    end_day = start_day.replace(year=start_day.year + 1)
                except ValueError:
                    end_day = (start_day + timedelta(1)).replace(year=start_day.year + 1)

                self.all_days = list(daterange(start_day, end_day))

                self.date_masks = [(d, self.date_mask(d)) for d in self.all_days]
                self.cached[self._today] = self.all_days, self.date_masks
        else:
            self.date_masks = None

    today = property(get_today, set_today, doc=u"""
    Устанавливает или снимает привязку к дате.
    Нужен для генереции дат хождений.
    """)

    def __str__(self):
        return "".join(self.mask & (1 << (self.MASK_LENGTH - 1 - d)) and '1' or '0'
                       for d in range(self.MASK_LENGTH))

    def __int__(self):
        return self.mask

    def __bool__(self):
        return self.mask != 0

    __nonzero__ = __bool__

    @classmethod
    def index(cls, date_):
        return (date_.month - 1) * 31 + (date_.day - 1)

    @classmethod
    def date_mask(cls, date_):
        # Год у нас начится с MSB, так как получаем маску мы с помщью int()
        return 1 << (cls.MASK_LENGTH - 1 - cls.index(date_))

    def __getitem__(self, date_):
        return bool(self.mask & self.date_mask(date_))

    def __setitem__(self, date_, runs):
        mask = self.date_mask(date_)

        if runs:
            self.mask |= mask
        else:
            self.mask = (self.mask | mask) ^ mask

    def __contains__(self, date_):
        return self[date_]

    def issubset(self, other):
        return self.mask & other.mask == self.mask

    def issuperset(self, other):
        return self.mask & other.mask == other.mask

    def difference(self, other):
        u"""Разность множеств дней хождений"""
        return self - other

    def __sub__(self, other):
        return (self | other) ^ other

    def __hash__(self):
        return hash((hash(self.today), self.mask))

    def __or__(self, other):
        return RunMask(self.mask | other.mask, today=self._today)

    def __and__(self, other):
        return RunMask(self.mask & other.mask, today=self._today)

    def __eq__(self, other):
        return self.mask == other.mask and self.today == other.today

    def __ne__(self, other):
        return not self.__eq__(other)

    def __xor__(self, other):
        return RunMask(self.mask ^ other.mask, today=self._today)

    def __repr__(self):
        if not self.today:
            self.today = date.today()
        dates = self.dates()
        days = u", ".join(d.strftime("%Y-%m-%d") for d in dates[:10])
        if len(dates) > 10:
            days += u"..."

        return u"RunMask[%s]" % days

    @classmethod
    def range(cls, start, end, today=None, include_end=False):
        mask = 0

        for d in daterange(start, end, include_end=include_end):
            mask |= cls.date_mask(d)

        return RunMask(mask, today=today)

    def portion(self, start_date, length):
        portion_mask = self.range(start_date, start_date + timedelta(length))

        return self & portion_mask

    def dates(self, past=True):
        u"""
        Список дат по которым ходит маршрут
        @param past: Брать ли дни раньше, чем self.today.
        """

        if self._today is None:
            raise ValueError("Method dates is not available on RunMask's without today")

        if past:
            return [d for d, d_mask in self.date_masks if self.mask & d_mask]
        else:
            return [d for d, d_mask in self.date_masks if self.mask & d_mask and d >= self._today]

    def iter_dates(self, past=True):
        u"""
        Список дат по которым ходит маршрут
        @param past: Брать ли дни раньше, чем self.today.
        """

        if self._today is None:
            raise ValueError("Method dates is not available on RunMask's without today")

        if past:
            return (d for d, d_mask in self.date_masks if self.mask & d_mask)
        else:
            return (d for d, d_mask in self.date_masks if self.mask & d_mask and d >= self._today)

    def shifted(self, shift):
        u"""
        Сдвигает маску на shift дней вперед
        """
        if shift:
            m = RunMask(today=self.today)

            for d in self.dates():
                m[d + timedelta(shift)] = 1
        else:
            m = RunMask(self, today=self.today)

        m.extrapolation_forbidden = self.extrapolation_forbidden

        return m

    _ranges = {}

    @classmethod
    def range_equal(cls, mask1, mask2, period_start, period_end):
        if (period_start, period_end) in cls._ranges:
            compare_range = cls._ranges[(period_start, period_end)]
        else:
            compare_range = RunMask.range(period_start, period_end)
            cls._ranges[(period_start, period_end)] = compare_range

        mask1 = mask1 & compare_range
        mask2 = mask2 & compare_range
        return mask1 == mask2

    @classmethod
    def mask_day_index(cls, dt):
        return (dt.day - 1) + (dt.month - 1) * 31

    @classmethod
    def runs_at(cls, year_days, dt):
        if not year_days:
            return False

        if len(year_days) != cls.MASK_LENGTH:
            log.error('Bad runmask %r' % year_days)
            return False

        return year_days[cls.mask_day_index(dt)] == '1'

    @classmethod
    def first_run(cls, year_days, today):
        u"""Вычисляет первый или последний день хождения"""

        year_days = str(year_days)

        if len(year_days) != cls.MASK_LENGTH:
            log.error('Bad runmask %r' % year_days)
            return None

        def safe_date(year, month, day):
            try:
                return date(year, month, day)
            except ValueError:
                assert (month == 2 and day == 29)

                return date(year, 3, 1)

        # Ищем первую дату хождения после сегодняшнего дня
        def first():
            day_position = (today.month - 1) * 31 + (today.day - 1)

            year = today.year

            try:
                first_day = year_days.index('1', day_position)
            except ValueError:
                year += 1

                try:
                    first_day = year_days.index('1')
                except ValueError:
                    return None

            month0, day0 = divmod(first_day, 31)

            return safe_date(year, month0 + 1, day0 + 1)

        def last():
            day_position = (today.month - 1) * 31 + (today.day - 1)

            year = today.year

            try:
                last_day = year_days.rindex('1', 0, day_position)
            except ValueError:
                year -= 1

                try:
                    last_day = year_days.rindex('1')
                except ValueError:
                    return None

            month0, day0 = divmod(last_day, 31)

            return safe_date(year, month0 + 1, day0 + 1)

        first_run = first()

        if first_run is None:
            return None

        if first_run - today > timedelta(365 - settings.DAYS_TO_PAST):
            first_run = last()

        return first_run

    def is_first_run_day(self, day):
        """
        Является ли заданая дата первым днем хождения нитки -
        первым среди всех известных нам дней вообще, а не относительно текущего дня.

        Т.к. в прошлом в маске содержится не более DAYS_TO_PAST дней, то смотрим только за этот период.

        :param day: datetime.date
        """
        shifted_day = day - timedelta(days=settings.DAYS_TO_PAST - 1)
        first_run_date = RunMask.first_run(str(self), shifted_day)

        return first_run_date == day

    def format_days_text(self, days_limit=True, shift=0, lang=None):
        lang = lang or translation.get_language()

        all_dates = self.dates()
        if not all_dates:
            return None

        if shift:
            all_dates = [d + timedelta(days=shift) for d in all_dates]

        if days_limit is None:
            # Не ограничиваем по датам и количеству
            return DaysText(
                group_days([(d.day, d.month) for d in all_dates], lang),
                False
            )

        dates_limit = self.today + timedelta(days=settings.DAYS_TO_FUTURE)
        limited_dates = list(
            takewhile(lambda d: d < dates_limit, all_dates)
        )

        if not limited_dates:
            limited_dates = all_dates

        if days_limit:
            # Даты хождения не в прошлом
            dates = list(
                dropwhile(lambda d: d < self.today, limited_dates)
            )

            # Если их меньше минимума, то берем просто минимум последних
            if len(dates) < DAYS_TEXT_LIMIT_MIN:
                dates = limited_dates[-DAYS_TEXT_LIMIT_MIN:]
        else:
            dates = limited_dates

        return DaysText(
            group_days([(d.day, d.month) for d in dates[:DAYS_TEXT_LIMIT_MAX]], lang),
            len(dates) > DAYS_TEXT_LIMIT_MAX
        )
