# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from builtins import zip
from builtins import map
import heapq
from bisect import bisect_right, bisect_left
from datetime import timedelta, time, datetime
from itertools import islice
from collections import defaultdict

import pytz
import six
from django.conf import settings

from travel.rasp.library.python.common23.date.run_mask import RunMask
from travel.rasp.library.python.common23.date import environment


class MaskSplitResult(object):
    def __init__(self, event_mask, result_mask, out_time, out_tz, shift):
        self.event_mask = event_mask
        self.result_mask = result_mask
        self.out_time = out_time
        self.out_tz = out_tz
        self.shift = shift


class Transformations(object):
    def __init__(self, transformations_by_event_time):
        self.transformations_by_event_time = transformations_by_event_time
        self.times = sorted(self.transformations_by_event_time.keys())

    def get_by_time(self, event_time):
        time_index = bisect_right(self.times, event_time) - 1
        base_event_time = self.times[time_index]

        return self.transformations_by_event_time[base_event_time]


class MaskSplitter(object):
    """
    Класс для разбиения маски на несколько, чтобы каждая из них переводилась из одной таймзоны в другую простым сдвигом
    Маску нельзя переводить между таймзонами простым сдвигом, если только в одной из них переводится время

    Предполагаем, что время маски, если попадает на переход с летнего на зимнее или обратно,
    указано после перехода.
    Т.е.
    Europe/Kiev +3/+2
    utc_transition
    +2
    2012-03-25 01:00 utc
    2012-03-25 03:10 kiv это уже летнее время после перехода.
    """

    def __init__(self):
        self._cache = {}

    def split(self, mask, event_time, event_tz, out_tz):
        if isinstance(event_tz, six.string_types):
            event_tz = pytz.timezone(event_tz)

        if isinstance(out_tz, six.string_types):
            out_tz = pytz.timezone(out_tz)

        naive_event_dt = datetime.combine(mask.today, event_time)
        transformations = self.build_transformations(mask.today, event_tz, out_tz)

        transformation = transformations.get_by_time(event_time)

        result = []

        # time_shift - сдвиг времени в секундах на любую дату
        # mask_shift - сдвиг для маски в event_tz, такой чтобы получить маску в out_tz
        # transform_mask - маска с датами отправления по event_tz, для которых
        #    мы будем отправляться в одно и тоже время в зоне out_tz
        for (time_shift, mask_shift), transform_mask in transformation.items():
            event_mask = transform_mask & mask
            if not event_mask:
                continue

            if mask_shift:
                result_mask = event_mask.shifted(mask_shift)
            else:
                result_mask = event_mask

            naive_out_dt = naive_event_dt + timedelta(seconds=time_shift)

            result.append(MaskSplitResult(
                event_mask, result_mask, naive_out_dt.time(), out_tz,
                mask_shift
            ))

        return result

    def _transition_iter(self, tz, start_dt):
        if not hasattr(tz, '_utc_transition_times'):
            return []

        start_index = bisect_left(tz._utc_transition_times, start_dt)
        return map(lambda tt: (tt, tz), islice(tz._utc_transition_times, start_index, None))

    def build_transformations(self, today, event_tz, out_tz):
        """
        Строим трансформации маски в зависимости от времени события в event_tz
        """
        key = (today, event_tz.zone, out_tz.zone)
        if key in self._cache:
            return self._cache[key]

        start_date = today - timedelta(settings.DAYS_TO_PAST)
        start_dt = datetime.combine(start_date, time(0, 0))
        last_date = today + timedelta(settings.MASK_SPLITTER_DAYS_TO_FUTURE)
        last_dt = datetime.combine(last_date, time(0, 0))

        # Собираем все даты в которые может случится переход на летнее/зимнее время
        # Собираем все времена, в которые может случится переход на летнее/зимнее время,
        #       либо переключение суток

        check_times = {time(0, 0)}
        switch_date_set = {start_date}

        transitions = heapq.merge(
            self._transition_iter(event_tz, start_dt),
            self._transition_iter(out_tz, start_dt)
        )

        for transition_dt, transition_tz in transitions:
            if transition_dt > last_dt:
                break

            transition_date = transition_dt.date()
            switch_date_set.update([
                transition_date - timedelta(2),
                transition_date - timedelta(1),
                transition_date,
                transition_date + timedelta(1),
                transition_date + timedelta(2)
            ])

            check_times.add(pytz.utc.localize(transition_dt).astimezone(event_tz).time())

        switch_dates = [_d for _d in sorted(switch_date_set) if start_date <= _d < last_date]
        for day in switch_dates:
            day_switch = out_tz.localize(datetime.combine(day, time(0, 0)))
            check_times.add(day_switch.astimezone(event_tz).time())

            next_day = day + timedelta(1)
            if next_day not in switch_date_set:
                day_switch = out_tz.localize(datetime.combine(next_day, time(0, 0)))
                check_times.add(day_switch.astimezone(event_tz).time())

        check_times = sorted(check_times)

        transformation_by_event_time = {}
        # Для каждого времени и для каждой даты проверяем какое время(out_time)
        # и какой сдвиг получается при переходе из event_tz в out_tz
        # Получаем маски сгруппированные по out_time_shift, out_shift
        for event_time in check_times:
            mask_by_out_time_shift_with_mask_shift = defaultdict(lambda: RunMask(today=today))

            event_dt = event_tz.localize(datetime.combine(switch_dates[0], event_time))
            out_dt = event_dt.astimezone(out_tz)
            out_time_shift = (
                out_dt.replace(tzinfo=None) - event_dt.replace(tzinfo=None)
            ).total_seconds()

            mask_shift = (out_dt.date() - event_dt.date()).days

            from_day = switch_dates[0]
            current_time_shift = out_time_shift
            current_mask_shift = mask_shift
            for day in switch_dates:
                event_dt = event_tz.localize(datetime.combine(day, event_time))
                out_dt = event_dt.astimezone(out_tz)
                out_time_shift = (
                    out_dt.replace(tzinfo=None) - event_dt.replace(tzinfo=None)
                ).total_seconds()
                mask_shift = (out_dt.date() - event_dt.date()).days

                if (out_time_shift, mask_shift) == (current_time_shift, current_mask_shift):
                    continue

                mask_by_out_time_shift_with_mask_shift[
                    (current_time_shift, current_mask_shift)
                ] |= RunMask.range(from_day, day, today=today, include_end=False)

                from_day = day
                current_time_shift = out_time_shift
                current_mask_shift = mask_shift

            mask_by_out_time_shift_with_mask_shift[
                (current_time_shift, current_mask_shift)
            ] |= RunMask.range(from_day, last_date, today=today, include_end=False)

            transformation_by_event_time[event_time] = mask_by_out_time_shift_with_mask_shift

        self._cache[key] = Transformations(transformation_by_event_time)

        return self._cache[key]


def mask_split(mask, event_time, event_tz, out_tz):
    """
    Разбивает маску на несколько, чтобы каждая из них переводилась из одной таймзоны в другую простым сдвигом
    Маску нельзя переводить между таймзонами простым сдвигом, если только в одной из них переводится время
    :param mask: Исходная маска
    :param event_time: Время события в исходной таймзоне, для которого сформирована маска
    :param event_tz: Исходная таймзона
    :param out_tz: Таймзона, в которую производится перевод
    :return:
    """
    splitter = MaskSplitter()
    return splitter.split(mask, event_time, event_tz, out_tz)


class StationForMaskSplit(object):
    """
    Станция для разбиения масок по таймзоне
    """
    def __init__(self, station_pytz, arrival_time, arrival_day_shift, departure_time, departure_day_shift):
        self.pytz = station_pytz
        self.arrival_time = arrival_time
        self.arrival_day_shift = arrival_day_shift
        self.departure_time = departure_time
        self.departure_day_shift = departure_day_shift


class ThreadForMaskSplit(object):
    """
    Подготовленная нитка для разбиения масок по таймзоне
    """
    def __init__(self, run_mask, stations):
        self.run_mask = run_mask
        self.stations = stations


def _update_mask_variants(mask_variants, new_mask):
    remainder_of_new_mask = new_mask
    updated_mask_variants = set()
    for existed_mask in mask_variants:
        intersection = existed_mask & remainder_of_new_mask
        if intersection:
            updated_mask_variants.add(intersection)

            remainder_of_existed_mask = existed_mask - intersection
            if remainder_of_existed_mask:
                updated_mask_variants.add(remainder_of_existed_mask)

            remainder_of_new_mask = remainder_of_new_mask - intersection
        else:
            updated_mask_variants.add(existed_mask)

    if remainder_of_new_mask:
        updated_mask_variants.add(remainder_of_new_mask)

    return updated_mask_variants


def thread_mask_split(thread_for_split, mask_splitter, out_pytz):
    """
    Разбивает маску дней хождения нитки, по некоторой указанной таймзоне
    :param thread_for_split: объект типа ThreadForMaskSplit
    :param mask_splitter: объект класса MaskSplitter
    :param out_pytz: таймзона, в которой времена отправления и прибытия должны быть одинаковыми для дней маски
    :return: список масок, на которые разбилась исходная
    """
    mask_variants = set()
    processed_mask_variants = set()

    for station_from, station_to in zip(thread_for_split.stations[:-1], thread_for_split.stations[1:]):
        if station_from.departure_time is None or station_to.arrival_time is None:
            continue

        from_mask = thread_for_split.run_mask.shifted(station_from.departure_day_shift)

        # Для каждого сегмента разбиваем маску по станциям отправления и прибытия
        for dep_split_res in mask_splitter.split(from_mask, station_from.departure_time, station_from.pytz, out_pytz):
            dep_mask_from = dep_split_res.event_mask
            arr_mask_from = dep_mask_from.shifted(station_to.arrival_day_shift - station_from.departure_day_shift)

            for arr_split_res in mask_splitter.split(arr_mask_from, station_to.arrival_time, station_to.pytz, out_pytz):
                arr_mask_both = arr_split_res.event_mask
                new_mask_variant = arr_mask_both.shifted(-station_to.arrival_day_shift)

                if new_mask_variant in processed_mask_variants:
                    continue

                mask_variants = _update_mask_variants(mask_variants, new_mask_variant)
                processed_mask_variants.add(new_mask_variant)
                processed_mask_variants.update(mask_variants)

    return sorted(mask_variants, key=lambda m: next(m.iter_dates()))


def calculate_mask_for_tz_and_station(init_mask, station_departure_time, station_days_shift, station_pytz, out_pytz):
    """
    Переводит маску дней хождения нитки в маску дней хождения на указанной станции в указанной таймзоне
    :param init_mask: исходная маска
    :param station_departure_time: время отправления со станции в таймзоне станции
    :param station_days_shift: сдвиг в днях между отправлением со станции и днем старта нитки
    :param station_pytz: таймзона станции
    :param out_pytz: таймзона, в которую надо перевести маску
    """
    result_mask = RunMask(today=environment.today())
    for d in init_mask.dates():
        naive_station_tz_dt = datetime.combine(d + timedelta(days=station_days_shift), station_departure_time)
        out_tz_dt = station_pytz.localize(naive_station_tz_dt).astimezone(out_pytz)
        shift = station_days_shift + (out_tz_dt.date() - naive_station_tz_dt.date()).days
        result_mask[d + timedelta(shift)] = 1
    return result_mask
