# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

from django.conf import settings

from travel.rasp.library.python.db.cluster_mdb import ClusterBase, ClusterMdbMysql

from travel.rasp.library.python.common23.data_api.mdb.instance import mdb_mysql_api
from travel.rasp.library.python.common23.db.backends.mysql.base import DatabaseWrapper as MysqlDatabaseWrapper
from travel.rasp.library.python.common23.db.backends.cluster_mdb.fallback_hosts_mds import FallbackHostsStorage
from travel.rasp.library.python.common23.settings.utils import define_setting


define_setting('DEFAULT_CLUSTER_INFO_TTL', default=60)

log = logging.getLogger(__name__)


class ClusterMdbMysqlWithFallbackUpdate(ClusterMdbMysql):
    def get_new_fallback_hosts(self):
        fallback_storage = FallbackHostsStorage(self.cluster_id)
        cluster_info = fallback_storage.get_cluster_info()

        return cluster_info['master'], cluster_info['hosts'], cluster_info.get('replicas_health')

    def update_fallback_hosts(self):
        log.info('updating fallback hosts for cluster %s', self.cluster_id)
        try:
            master_host, all_hosts, replicas_health = self.get_new_fallback_hosts()
        except Exception:
            log.exception("Unable to get fallback hosts for cluster %s", self.cluster_id)
        else:
            self.set_fallback_hosts(master_host, all_hosts)
            self.set_replicas_health(replicas_health)

    def on_before_update_cluster_configuration(self):
        # type: (ClusterMdbMysqlWithFallbackUpdate) -> None
        self.update_fallback_hosts()
        super(ClusterMdbMysqlWithFallbackUpdate, self).on_before_update_cluster_configuration()


class DatabaseWrapper(MysqlDatabaseWrapper):
    """ Бэкенд для работы с кластером Mysql MDB"""
    def create_cluster(self):
        # type: () -> ClusterBase
        kwargs = self.get_cluster_kwargs()
        kwargs.update(dict(
            mdb_client=mdb_mysql_api,
            cluster_id=self.cluster_settings['CLUSTER_ID'],
            mdb_api_call_enabled=self.cluster_settings.get('MDB_API_CALL_ENABLED', True),
            check_master_on_each_connect=self.cluster_settings.get('CHECK_MASTER_ON_EACH_CONNECT', False),
            cluster_info_ttl=self.cluster_settings.get('CLUSTER_INFO_TTL', int(settings.DEFAULT_CLUSTER_INFO_TTL)),
        ))

        fallback = self.cluster_settings.get('FALLBACK')
        if fallback:
            kwargs.update(dict(
                fallback_master=fallback['MASTER'],
                fallback_replicas=fallback['REPLICAS'],
            ))

        return ClusterMdbMysqlWithFallbackUpdate(**kwargs)
