# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import json
import logging
from datetime import datetime

from django.conf import settings
from retry import retry

from travel.library.python.tracing.instrumentation import traced_function
from travel.rasp.library.python.db.replica_health import ReplicaHealth

from travel.rasp.library.python.common23.data_api.juggler.juggler import Event, OK, WARN, send_events
from travel.rasp.library.python.common23.data_api.mdb.instance import mdb_mysql_api
from travel.rasp.library.python.common23.db.mds.clients import mds_s3_infra_client
from travel.rasp.library.python.common23.db.replica_checker import DefaultReplicaChecker
from travel.rasp.library.python.common23.settings.utils import define_setting


define_setting("MDS_PATH_MDB_CLUSTER_INFO", default="mdb/clusters/")

log = logging.getLogger(__name__)


class FallbackHostsStorage(object):
    def __init__(self, cluster_id, mds_client=None):
        self.cluster_id = cluster_id
        self.cluster_info_path = settings.MDS_PATH_MDB_CLUSTER_INFO + self.cluster_id
        self.client = mds_client or mds_s3_infra_client

    @retry(tries=2, delay=0.1)
    @traced_function
    def get_cluster_info(self):
        data_stream = self.client.get_data(self.cluster_info_path)
        return json.load(data_stream)

    @retry(tries=2, delay=1)
    @traced_function
    def set_cluster_info(self, data):
        data_str = json.dumps(data, indent=4)
        self.client.save_data(self.cluster_info_path, data_str)


@retry(tries=2, delay=3)
def update_mdb_cluster_info_in_mds(cluster_id, cluster_name, replica_checker):
    log.info("Updating cluster info in mds for cluster %s (%s)", cluster_id, cluster_name)
    cluster_info = mdb_mysql_api.get_cluster_info(cluster_id)
    data = {
        'updated': datetime.utcnow().isoformat(),
        'hosts': [inst.hostname for inst in cluster_info.instances],
        'replicas_health': dict([get_replica(inst, replica_checker) for inst in cluster_info.instances]),
        'master': cluster_info.master.hostname,
        'cluster_id': cluster_info.cluster_id,
        'cluster_name': cluster_name
    }
    hosts_storage = FallbackHostsStorage(cluster_id)
    hosts_storage.set_cluster_info(data)

    try:
        send_events(collect_events(data), source='update_mdb_cluster_info_in_mds')
    except Exception as ex:
        log.info('Error while sending events: {}'.format(ex))


def collect_events(data):
    for host, health in data['replicas_health'].items():
        yield Event(
            service='mysql_replica_health',
            host=host,
            status=OK if health == ReplicaHealth.ALIVE else WARN,
            description='Replica health: {}'.format(health),
            tags=['mysql']
        )


def get_replica(inst, replica_checker):
    host = inst.hostname
    health = inst.raw_data.get('health', ReplicaHealth.UNKNOWN)

    if inst.is_master:
        return host, health

    # не производим собственных проверок, если mdb не считает реплику живой
    if health == ReplicaHealth.ALIVE and not replica_checker.check_replica(host):
        health = ReplicaHealth.SUSPENDED
        log.warn('Replica %s is suspended', host)

    return host, health


def update_mdb_clusters_info_in_mds(clusters):
    log.info("Updating clusters info for clusters: %s", clusters)
    for cluster in clusters:
        try:
            update_mdb_cluster_info_in_mds(
                cluster['cluster_id'],
                cluster['cluster_name'],
                cluster.get('replica_checker', DefaultReplicaChecker()))

        except Exception:
            log.exception("Can't update info for cluster %s (%s)", cluster['cluster_id'], cluster['cluster_name'])
    log.info("Updating clusters info done")
