# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from builtins import map
import logging
import time as c_time
import traceback
from threading import local

from django.conf import settings
from django.db.backends.mysql.base import DatabaseWrapper as MysqlDatabaseWrapper, Database
from django.db.backends.utils import CursorWrapper
from django.utils.encoding import force_text

from travel.rasp.library.python.common23.db import connect
from travel.rasp.library.python.common23.settings.utils import define_setting, bool_converter


log = logging.getLogger(__name__)

define_setting('MYSQL_LOG_QUERIES', default=False, converter=bool_converter)


class CursorLoggingWrapper(CursorWrapper):
    storage = local()

    def add_time(self, duration):
        try:
            self.storage.cumulative_time += duration
        except AttributeError:
            self.storage.cumulative_time = duration

    def execute(self, sql, params=None):
        start = c_time.time()
        try:
            return super(CursorLoggingWrapper, self).execute(sql, params)
        finally:
            duration = c_time.time() - start
            sql = self.db.ops.last_executed_query(self.cursor, sql, params)
            log.debug('(%.3f) %s; args=%s' % (duration, sql, params),
                      extra={'duration': duration, 'sql': sql, 'params': params})
            self.add_time(duration)

    def executemany(self, sql, param_list):
        start = c_time.time()
        try:
            return super(CursorLoggingWrapper, self).executemany(sql, param_list)
        finally:
            duration = c_time.time() - start
            log.debug('(%.3f) %s; args=%s' % (duration, sql, param_list),
                      extra={'duration': duration, 'sql': sql, 'params': param_list})
            self.add_time(duration)

    @classmethod
    def get_cumulative_time(cls):
        return getattr(cls.storage, 'cumulative_time', 0)

    @classmethod
    def get_cumulative_time_and_reset(cls):
        ct = cls.get_cumulative_time()
        cls.storage.cumulative_time = 0
        return ct


class DatabaseWrapper(MysqlDatabaseWrapper):
    """ Обычный MysqlWrapper, но с нашими настройками. """
    isolation_level = None

    def get_new_connection(self, conn_params):
        return connect.get_connection(self.settings_dict)

    def get_connection_params(self):
        """ Нам не нужна логика MysqlDatabaseWrapper'а, т.к. настройки соединения мы формируем сами. """
        return self.settings_dict.copy()

    def is_usable(self):
        try:
            self.connection.ping()
        except Database.Error:
            log.exception('Ошибка соединения с базой.\n%s\n',
                          ''.join(map(force_text, traceback.format_stack())))
            return False
        else:
            return True

    def ensure_connection(self):
        if settings.PING_MYSQL_BEFORE_EACH_REQUEST and self.connection and not self.is_usable():
            log.info('Принудительно закрываем соединение с базой')
            self.close()

        super(DatabaseWrapper, self).ensure_connection()

    def get_db_name(self):
        self.ensure_connection()
        return self.connection.conn_params['db']

    def cursor(self):
        cursor = super(DatabaseWrapper, self).cursor()
        need_to_log = settings.MYSQL_LOG_QUERIES
        return CursorLoggingWrapper(cursor, self) if need_to_log else cursor
