# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

import MySQLdb
import typing

from django.conf import settings

from travel.rasp.library.python.db.cluster import ClusterConst, DbInstance, ConnectionFailure, ClusterBase

from travel.rasp.library.python.common23.db import connect
from travel.rasp.library.python.common23.db.backends.dbwrapper_base import DatabaseWrapper as BaseDatabaseWrapper


log = logging.getLogger(__name__)


class DatabaseWrapper(BaseDatabaseWrapper):
    def __init__(self, *args, **kwargs):
        super(DatabaseWrapper, self).__init__(*args, **kwargs)
        self.cluster_settings = self.settings_dict.get('CLUSTER', {})
        self.cluster = self.create_cluster()
        self.current_dc = settings.YANDEX_DATA_CENTER

    def create_cluster(self):
        # type: () -> ClusterBase
        kwargs = self.get_cluster_kwargs()
        hosts = self.cluster_settings.get('HOSTS')
        if not hosts:
            hosts = [self.settings_dict['HOST']]
        instances = [DbInstance(host=host, is_master=(i == 0)) for i, host in enumerate(hosts)]

        return ClusterConst(instances=instances, **kwargs)

    def get_cluster_kwargs(self):
        """ Дефолтные параметры для ClusterBase """

        use_replicas = self.cluster_settings.get('USE_REPLICAS', True)
        use_master = self.cluster_settings.get('USE_MASTER', True)
        instance_filter = choose_cluster_instance_filter(use_replicas, use_master)

        return dict(
            connection_getter=self.get_connection_to_instance,
            instance_filter=instance_filter,
            instance_dead_ttl=self.cluster_settings.get('INSTANCE_DEAD_TTL', 30),
        )

    def get_cluster(self):
        return self.cluster

    def get_new_connection(self, conn_params):
        return self.cluster.get_connection(current_dc=self.current_dc)

    def get_connection_to_instance(self, inst):
        # type: (DbInstance) -> typing.Any
        return self.get_connection_to_host(inst.host)

    def get_connection_to_host(self, host):
        """
        Непосредственно получение коннекта на заданный хост.
        Функция передается как параметр connection_getter в инстанс ClusterBase, где используется для создания соединений
        """
        try:
            return connect.get_connection(self.settings_dict, host)
        except MySQLdb.OperationalError as ex:
            raise ConnectionFailure(repr(ex))

    def get_hosts(self):
        return [inst.host for inst in self.cluster.get_instances()]

    def get_all_hosts(self):
        return [inst.host for inst in self.cluster.instances]


def inst_filter_replica(inst):
    return inst.is_replica


def inst_filter_master(inst):
    return inst.is_master


def inst_filter_any(inst):
    return True


def choose_cluster_instance_filter(use_replicas, use_master):
    assert use_replicas or use_master

    if use_replicas:
        if use_master:
            instance_filter = inst_filter_any
        else:
            instance_filter = inst_filter_replica
    else:
        instance_filter = inst_filter_master

    return instance_filter
