# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from future import standard_library
standard_library.install_aliases()
from functools import wraps
from threading import RLock

from travel.rasp.library.python.common23.db.switcher import switcher


import logging
from multiprocessing.dummy import Pool as ThreadPool
from threading import Thread

from django.core.cache import caches
from django.conf import settings

from travel.rasp.library.python.common23.settings.utils import define_setting


log = logging.getLogger(__name__)


define_setting('DEFAULT_CACHE_SYNC_TIMEOUT', default=4., converter=float)


def cache_until_switch(func):
    """
    Кеширует результаты выполнения функции, обновляет результаты если сменилась база.
    Все аргументы должны быть hashable.
    """
    results = {}

    def clean_cache(**kwargs):
        for k in results.keys():
            del results[k]
    switcher.data_updated.connect(clean_cache, weak=False)

    @wraps(func)
    def new_func(*args, **kwargs):
        key = args + tuple([(k, v) for k, v in kwargs.items()])
        try:
            return results[key]
        except KeyError:
            results[key] = func(*args, **kwargs)
            return results[key]

    return new_func


def cache_until_switch_thread_safe(func):
    """
    Кеширует результаты выполнения функции, обновляет результаты если сменилась база.
    Все аргументы должны быть hashable. tf значит thread safe
    """
    results = {}
    global_lock = RLock()
    locks = {}

    def get_key_lock(key):
        try:
            return locks[key]
        except KeyError:
            locks[key] = RLock()
            return locks[key]

    def clean_cache(**kwargs):
        try:
            global_lock.acquire()
            try:
                for l in locks.values():
                    l.acquire()
                for k in results.keys():
                    del results[k]
            finally:
                for l in locks.values():
                    l.release()
        finally:
            global_lock.release()
    switcher.data_updated.connect(clean_cache, weak=False)

    @wraps(func)
    def new_func(*args, **kwargs):
        try:
            global_lock.acquire()
            key = args + tuple([(k, v) for k, v in kwargs.items()])
            key_lock = get_key_lock(key)
        finally:
            global_lock.release()

        try:
            key_lock.acquire()
            try:
                return results[key]
            except KeyError:
                results[key] = func(*args, **kwargs)
                return results[key]
        finally:
            key_lock.release()

    return new_func


def _call_method_on_cache_alias(alias, name, *args, **kwargs):
    try:
        cache = caches[alias]
    except Exception:
        log.exception('Ошибка при открытии кэша %s в потоке', alias)
        return

    try:
        method = getattr(cache, name)
        method(*args, **kwargs)
    except Exception:
        log.exception('Ошибка при вызове метода %s кэша %s в потоке', name, alias)

    cache.close()


def _get_global_cache_aliases():
    """
    'default' - кэш в нашем ДЦ, есть еще кэши в других ДЦ, например, 'iva'.
    Так как у нас есть еще и локальные кэши, в которые мы здесь ничего
    сохранять не хотим - отфильтровываем их.
    """
    if getattr(settings, 'ENABLE_REDIS_CACHE', False):
        # У redis один кластер (с репликацией) - default.
        return ['default']

    aliases = []
    for alias, params in settings.CACHES.items():
        if params['BACKEND'].startswith('travel.rasp.library.python.common23.db.memcache_backend'):
            aliases.append(alias)
    return aliases


def global_cache_methodcall(name, *args, **kwargs):
    aliases = _get_global_cache_aliases()
    for alias in aliases:
        thread = Thread(target=_call_method_on_cache_alias, args=(alias, name) + args, kwargs=kwargs)
        thread.start()


def global_cache_add(key, data, timeout=None):
    global_cache_methodcall('add', key, data, timeout)


def global_cache_set(key, data, timeout=None):
    global_cache_methodcall('set', key, data, timeout)


def global_cache_delete(key):
    global_cache_methodcall('delete', key)


def global_cache_sync_methodcall(name, *args, **kwargs):
    """
    wait_for_sync_timeout максимальное время ожидания операции во всех thread,
    будет использован только если передан в качестве именованного аргумента.
    Важно! После истечения таймаута недоработавшие thread продолжают работать до завершения операции.

    Нужно как-то запустить параллельно работу с несколькими мемкэшами.
    При этом нужно:
    1. отдавать ответ пользователю по истечении таймаута (или раньше);
    2. корректно завершать работу ThreadPool.

    Решение.
    Использовать thread, который потом ждать с таймаутом, после таймаута отдавать ответ,
    а thread пусть работает дальше. Внутри этого thread создавать пул, который работает с мемкэшами.
    В thread дождаться (синхронно) окончания работы всего пула и делать terminate().
    В этом варианте за ожидание клиентом не больше таймаута отвечает код запустивший thread,
    а за удаление пула отвечает thread.
    """
    sync_timeout = kwargs.pop('wait_for_sync_timeout', settings.DEFAULT_CACHE_SYNC_TIMEOUT)
    aliases = _get_global_cache_aliases()

    def do_work_in_pool():
        pool = ThreadPool(len(aliases))
        try:
            pool.map(
                lambda alias: _call_method_on_cache_alias(alias, name, *args, **kwargs),
                aliases
            )
        finally:
            pool.terminate()

    thread = Thread(target=do_work_in_pool)
    thread.start()
    thread.join(timeout=sync_timeout)
    if thread.is_alive():
        log.warning('Глобальная memcached операция %s не успела завершиться за %s секунд', name, sync_timeout)


def global_cache_sync_add(key, data, timeout=None, wait_for_sync_timeout=settings.DEFAULT_CACHE_SYNC_TIMEOUT):
    global_cache_sync_methodcall('add', key, data, timeout, wait_for_sync_timeout=wait_for_sync_timeout)


def global_cache_sync_set(key, data, timeout=None, wait_for_sync_timeout=settings.DEFAULT_CACHE_SYNC_TIMEOUT):
    global_cache_sync_methodcall('set', key, data, timeout, wait_for_sync_timeout=wait_for_sync_timeout)


def global_cache_sync_delete(key, wait_for_sync_timeout=settings.DEFAULT_CACHE_SYNC_TIMEOUT):
    global_cache_sync_methodcall('delete', key, wait_for_sync_timeout=wait_for_sync_timeout)


def get_package_cache_root():
    return '/yandex/rasp/{}/'.format(settings.PKG_VERSION)
