# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import typing  # noqa
from copy import copy

from django.conf import settings
from django.db.backends.mysql.base import Database, CLIENT, django_conversions
from django.utils import six
from django.utils.encoding import force_str
from django.utils.safestring import SafeBytes, SafeText
from MySQLdb.constants import FIELD_TYPE, FLAG


log = logging.getLogger(__name__)


def string_decode(s):
    return s.decode('utf-8')


rasp_conv = django_conversions.copy()
rasp_conv.update({
    FIELD_TYPE.STRING: [(None, string_decode)],
    FIELD_TYPE.VAR_STRING: [(None, string_decode)],
    FIELD_TYPE.VARCHAR: [(None, string_decode)],
    FIELD_TYPE.BLOB: ((FLAG.BINARY, None), (None, string_decode))
})


CONNECT_OPTIONS = {
    'charset': 'utf8',
    'use_unicode': False,
    'local_infile': True,
    'client_flag': CLIENT.FOUND_ROWS,
    'conv': rasp_conv,
    'connect_timeout': settings.MYSQL_CONNECT_TIMEOUT,
    'read_timeout': settings.MYSQL_READ_TIMEOUT,
    'write_timeout': settings.MYSQL_WRITE_TIMEOUT,
}


def get_params(settings_dict, host=None, db_name=None):
    """
    Получение параметров коннекта к mysql в формате MySQLdb.connect из django-like настроек.

    Сделано на основе логики и настроек из django.backends.mysql.base.DatabaseWrapper.
    """
    conn_params = copy(CONNECT_OPTIONS)

    if not host:
        host = settings_dict['HOST']

    if host.startswith('/'):
        conn_params['unix_socket'] = host
    elif host:
        conn_params['host'] = host

    if settings_dict.get('PORT'):
        conn_params['port'] = int(settings_dict['PORT'])

    if not db_name:
        db_name = settings_dict.get('NAME')
    conn_params['db'] = db_name

    if settings_dict.get('USER') is not None:
        conn_params['user'] = settings_dict['USER']

    if settings_dict.get('PASSWORD') is not None:
        conn_params['passwd'] = force_str(settings_dict['PASSWORD'])

    conn_params.update(settings_dict.get('OPTIONS', {}))

    return conn_params


class ConnectionWrapper(object):
    """ Враппер для хранения дополнительной информации о соединении. """

    def __init__(self, connection, conn_params):
        self.connection = connection
        self.conn_params = conn_params

    def __getattr__(self, item):
        return getattr(self.connection, item)

    def __repr__(self):
        return '{}->{}'.format(self.conn_params['host'], self.conn_params['db'])


def get_connection(settings_dict, host=None, db_name=None):
    # type: (dict, typing.Optional[str], typing.Optional[str]) -> ConnectionWrapper
    """
    Создание соединения с mysql.
    Сделано на основе логики и настроек из django.backends.mysql.base.DatabaseWrapper.
    """

    conn_params = get_params(settings_dict, host, db_name)
    conn = Database.connect(**conn_params)

    conn.encoders[SafeText] = conn.encoders[six.text_type]
    conn.encoders[SafeBytes] = conn.encoders[bytes]

    conn_wrapper = ConnectionWrapper(conn, conn_params)

    log.debug('connected to: %s', conn_wrapper)

    return conn_wrapper
