# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os
from fnmatch import fnmatch

import boto3
from botocore.client import Config
from django.conf import settings

from travel.library.python.tracing.instrumentation import traced_function

import travel.rasp.library.python.common23.db.mds.settings  # noqa
from travel.rasp.library.python.common23.utils.iterrecipes import chunker

log = logging.getLogger(__name__)


class MDSS3Wrapper(object):
    def __init__(self, bucket, url=settings.MDS_URL, key=settings.MDS_ACCESS_KEY_ID,
                 secret_key=settings.MDS_ACCESS_SECRET_KEY):
        self.bucket = bucket
        self.url = url
        self.key = key
        self.secret_key = secret_key
        self._client = None

    @property
    def client(self):
        if not self._client:
            self._client = get_mds_s3_raw_client(self.url, self.key, self.secret_key)

        return self._client

    @traced_function
    def save_data(self, key, data):
        if not settings.MDS_ENABLE_WRITING:
            log.error('Writing to MDS is not allowed in this environment')
            return

        return self.client.put_object(
            Bucket=self.bucket,
            Key=key,
            Body=data,
        )

    @traced_function
    def get_data(self, key):
        obj = self.client.get_object(Bucket=self.bucket, Key=key)

        return obj['Body']

    def __getattr__(self, attr_name):
        return getattr(self.client, attr_name)

    def upload_directory(self, dirpath, mds_base_path, ignore_files=None, ignore_case=False,
                         skip_empty_files=True):
        """
        :param dirpath:
        :param mds_base_path:
        :param ignore_files: fnmatch patterns
        :param skip_empty_files: botocore загружает пустые файлы с ошибкой, фикс не тривиален, потому костыль :(
        :return:
        """
        if not settings.MDS_ENABLE_WRITING:
            log.error('Writing to MDS is not allowed in this environment')
            return

        ignore_files = ignore_files or []
        dirpath = os.path.abspath(dirpath)
        log.info('Загружаем %s в %s', dirpath, mds_base_path)

        def match_ignores(fname, base_dir):
            for pattern in ignore_files:
                if pattern.startswith('/'):
                    # do absolute match
                    pattern = pattern[1:]
                    fpath = os.path.relpath(os.path.join(base_dir, fname), dirpath)
                else:
                    fpath = fname

                if ignore_case:
                    fpath = fpath.lower()
                    pattern = pattern.lower()
                if fnmatch(fpath, pattern):
                    return True
            else:
                return False

        for base_dir, _dirs, files in os.walk(dirpath):
            for fname in files:
                if match_ignores(fname, base_dir):
                    log.debug('Skip file - ignore list: %s %s', fname, base_dir)
                    continue

                filepath = os.path.abspath(os.path.join(base_dir, fname))
                if not os.path.isfile(filepath):
                    log.debug('Skip file - not a file: %s', filepath)
                    continue

                if skip_empty_files and os.stat(filepath).st_size == 0:
                    log.debug('Skip file - zero size: %s', filepath)
                    continue

                key = os.path.join(mds_base_path, os.path.relpath(filepath, dirpath))
                log.info('Uploading file %s to bucket {%s}, path {%s}', filepath, self.bucket, key)
                self.client.upload_file(Filename=filepath, Bucket=self.bucket, Key=key)

        log.info('Загрузили %s в %s', dirpath, mds_base_path)

    def download_directory(self, prefix, directory_path, only_new=False, remove_base_path=None):
        for obj in self.get_prefix_keys(prefix=prefix):
            key = obj['Key']
            if not remove_base_path:
                dir_key = os.path.join(directory_path, key)
            else:
                dir_key = os.path.join(directory_path, key[len(remove_base_path):].lstrip('/'))

            if only_new:
                if os.path.exists(dir_key):
                    continue

            if not os.path.exists(os.path.dirname(dir_key)):
                os.makedirs(os.path.dirname(dir_key))
            self.client.download_file(self.bucket, key, dir_key)

    def get_prefix_keys_by_part(self, prefix):
        next_token = ''
        kwargs = {
            'Bucket': self.bucket,
            'Prefix': prefix
        }

        while next_token is not None:
            if next_token != '':
                kwargs.update({'ContinuationToken': next_token})

            res = self.client.list_objects_v2(**kwargs)
            next_token = res.get('NextContinuationToken')

            yield res['Contents']

    def get_prefix_keys(self, prefix):
        for keys_part in self.get_prefix_keys_by_part(prefix):
            for obj in keys_part:
                yield obj

    def delete_prefix_keys(self, prefix):
        for keys_part in self.get_prefix_keys_by_part(prefix):
            objs = []
            for obj in keys_part:
                objs.append({
                    'Key': obj['Key']
                })

            # request contains a list of up to 1000 keys that you want to delete
            self.client.delete_objects(Bucket=self.bucket, Delete={'Objects': objs})

    def delete_keys(self, objs_keys, chunk_size=1000):
        deleted_keys = []

        for objs_keys_chunk in chunker(objs_keys, chunk_size=chunk_size):
            result = self.client.delete_objects(Bucket=self.bucket, Delete={'Objects': objs_keys_chunk})
            deleted_keys.extend(result['Deleted'])

        return deleted_keys


def get_mds_s3_raw_client(url=settings.MDS_URL, key=settings.MDS_ACCESS_KEY_ID, secret_key=settings.MDS_ACCESS_SECRET_KEY):
    session = boto3.session.Session(
        aws_access_key_id=key,
        aws_secret_access_key=secret_key,
    )

    config = Config(
        connect_timeout=settings.MDS_CONNECT_TIMEOUT,
        read_timeout=settings.MDS_READ_TIMEOUT,
        retries={'max_attempts': settings.MDS_RETRIES_MAX_ATTEMPTS}
    )

    return session.client(
        service_name='s3',
        endpoint_url=url,
        config=config,
    )
