# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

import six

if six.PY2:
    import pathlib2 as pathlib
else:
    import pathlib

import pytz
from django.conf import settings

import travel.rasp.library.python.common23.db.mds.settings  # noqa
from travel.rasp.library.python.common23.db.mds.base import get_mds_s3_raw_client
from travel.rasp.library.python.common23.utils.files.fileutils import get_file_modify_dt
from travel.rasp.library.python.common23.utils_db.threadutils import Ticker


log = logging.getLogger(__name__)


class MdsSyncer(Ticker):
    def __init__(self, config, interval, *args, **kwargs):
        """
        config: [
            {
                'local_path': str,
                'mds_key': str,
                'mds_bucket': <str> (not required, default is settings.MDS_RASP_COMMON_BUCKET)
            },
            ...
        ]
        """
        super(MdsSyncer, self).__init__(target=self.update_files, interval=interval, *args, **kwargs)
        self.config = config

        self.mds_client = get_mds_s3_raw_client()

    def update_files(self):
        for conf in self.config:
            try:
                self.update_file(conf)
            except Exception:
                log.exception("Can't update file %s", conf)

    def update_file(self, conf):
        local_path, mds_key = conf['local_path'], conf['mds_key']
        mds_bucket = conf.get('mds_bucket', settings.MDS_RASP_COMMON_BUCKET)
        if self._is_file_modified(mds_bucket, mds_key, local_path):
            log.info(
                'File %s is outdated, downloading from %s (%s %s)',
                local_path, settings.MDS_URL, mds_bucket, mds_key
            )
            pathlib.Path(local_path).parent.mkdir(parents=True, exist_ok=True)
            self.mds_client.download_file(mds_bucket, mds_key, local_path)

    def _is_file_modified(self, mds_bucket, mds_key, local_path):
        """
        Check if local file is older (or doesn't exist) than in mds.
        """

        # mds returns dt in utc
        modified_local = get_file_modify_dt(local_path)
        if not modified_local:  # no file
            return True

        modified_local = modified_local.replace(microsecond=0)  # mds rounds dt to seconds, so should we
        meta = self.mds_client.head_object(Bucket=mds_bucket, Key=mds_key)

        return meta['LastModified'] > pytz.UTC.localize(modified_local)
