# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import inspect
import logging

from django.apps import apps
from mongoengine import Document, EmbeddedDocument
from mongoengine.base import BaseDocument

from travel.rasp.library.python.common23.date.environment import is_migration_allowed

log = logging.getLogger(__name__)


class RaspDocument(Document):
    meta = {
        'abstract': True,
        'auto_create_index': False
    }


class MongoengineDeclarationError(Exception):
    pass


def ensure_indexes(module, additional_allowed_base_classes=()):
    for cls in additional_allowed_base_classes:
        if not issubclass(cls, BaseDocument):
            raise MongoengineDeclarationError(
                'Additional allowed class {} must be subclass of BaseDocument'.format(cls))
        if cls._meta.get('auto_create_index', True):
            raise MongoengineDeclarationError(
                'Additional allowed class {} must set auto_create_index to False'.format(cls))

    for name, cls in inspect.getmembers(module, predicate=lambda c: inspect.isclass(c) and issubclass(c, BaseDocument)):
        if (
            cls is not Document
            and not issubclass(cls, (RaspDocument, EmbeddedDocument) + additional_allowed_base_classes)
        ):
            raise MongoengineDeclarationError('Document must be declared as a subclass of {}: {}'.format(
                (RaspDocument,) + additional_allowed_base_classes, cls
            ))

        if is_migration_allowed():
            if issubclass(cls, (RaspDocument,) + additional_allowed_base_classes) and not cls._meta.get('abstract'):
                log.info('Begin ensure indexes for %s', cls)
                cls.ensure_indexes()
                log.info('Done ensure indexes for %s', cls)


def ensure_indexes_in_installed_apps():
    for app in apps.get_app_configs():
        if hasattr(app, 'ensure_mongo_indexes'):
            try:
                app.ensure_mongo_indexes()
            except Exception:
                log.exception('Ensure indexes error in %s', app)
