# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from django.conf import settings
from mongoengine import connection, register_connection

from travel.rasp.library.python.common23.db.mongo.command_logger import CommandLogger
from travel.rasp.library.python.common23.settings.utils import define_setting, bool_converter


log = logging.getLogger(__name__)

DEFAULT_MONGO_PORT = 27017


def _get_hosts_list(conf):
    try:
        if 'dbaas_id' in conf:
            from travel.rasp.library.python.common23.data_api.dbaas.instance import dbaas_client
            from travel.rasp.library.python.common23.data_api.dbaas.client import DbType
            hosts = [host.host for host in dbaas_client.get_hosts(conf['dbaas_id'], DbType.MONGODB)]
        else:
            hosts = conf.get('host', 'localhost')
            if not isinstance(hosts, (list, tuple)):
                hosts = [hosts]
    except Exception:
        # use hardcoded hosts list in case something went wrong
        fallback_hosts = conf.get('fallback_hosts')
        if fallback_hosts:
            hosts = fallback_hosts
        else:
            raise

    return hosts


def get_conn_string(conf):
    hosts = _get_hosts_list(conf)

    hosts_str = ','.join('{}:{}'.format(host, conf.get('port', DEFAULT_MONGO_PORT)) for host in hosts)
    return 'mongodb://' + hosts_str


def register_mongo_connections():
    define_setting('MONGO_LOG_COMMANDS', default=False, converter=bool_converter)

    for alias, conf in settings.MONGO_DATABASES.items():
        kwargs = {
            'name': conf['db'],
            'host': get_conn_string(conf),
            'username': conf.get('user'),
            'password': conf.get('password')
        }
        kwargs.update(conf.get('options', {}))
        if settings.MONGO_LOG_COMMANDS:
            kwargs['event_listeners'] = [CommandLogger()]
        register_connection(alias, **kwargs)


class DatabaseError(Exception):
    pass


class Databases(object):
    def __init__(self):
        self.databases = {}

    def __getitem__(self, db_alias):
        if db_alias not in settings.MONGO_DATABASES:
            raise DatabaseError('No such db alias: {}'.format(db_alias))

        if db_alias not in self.databases:
            self.databases[db_alias] = ConnectionProxy(db_alias)

        return self.databases[db_alias]


class ConnectionProxy(object):
    connections_registry = {}

    def __init__(self, alias):
        self.alias = alias

    def __getattr__(self, item):
        return getattr(self.get_connection(self.alias), item)

    def __repr__(self):
        return repr(self.get_connection(self.alias))

    @classmethod
    def get_connection(cls, alias):
        if alias not in cls.connections_registry:
            cls.connections_registry[alias] = connection.get_db(alias)
        return cls.connections_registry[alias]

    @classmethod
    def close_connections(cls):
        for alias, db in cls.connections_registry.items():
            log.info('Closing MongoClient for alias %s', alias)
            db.client.close()
            cls.connections_registry.pop(alias)


databases = Databases()
database = ConnectionProxy('default')
