# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import socket

from django.conf import settings
from django.core.cache.backends.base import DEFAULT_TIMEOUT
from django_redis.client import DefaultClient
from django_redis.exceptions import ConnectionInterrupted
from redis.sentinel import Sentinel

from travel.rasp.library.python.common23.settings.utils import define_setting, bool_converter


log = logging.getLogger(__name__)


define_setting('REDIS_FILTER_BY_DC', default=True, converter=bool_converter)


class DjangoRedisSentinel(Sentinel):
    def __init__(self, *args, **kwargs):
        self.current_dc = kwargs.pop('current_dc')
        super(DjangoRedisSentinel, self).__init__(*args, **kwargs)
        self.dc_by_ip = {}

    def update_slaves_dc(self, slaves):
        for ip, port in slaves:
            if ip not in self.dc_by_ip:
                try:
                    self.dc_by_ip[ip] = socket.gethostbyaddr(ip)[0].split('-')[0]
                except Exception:
                    log.exception('update_slaves_dc failed. ip: {}'.format(ip))

    def filter_slaves(self, slaves):
        """
        У реплик нет приоритетов, они распределяются через Round-robin.
        https://a.yandex-team.ru/arc/trunk/arcadia/contrib/python/redis/redis/sentinel.py?rev=r6902469#L118
        https://a.yandex-team.ru/arc/trunk/arcadia/contrib/python/redis/redis/sentinel.py?rev=r6902469#L237
        https://a.yandex-team.ru/arc/trunk/arcadia/contrib/python/redis/redis/connection.py?rev=r6902469#L1192
        Поэтому в случае наличия реплик в текущем ДЦ оставляем только их.
        """
        slaves_alive = super(DjangoRedisSentinel, self).filter_slaves(slaves)

        if not settings.REDIS_FILTER_BY_DC:
            return slaves_alive
        else:
            self.update_slaves_dc(slaves_alive)
            by_dc = [(ip, port) for ip, port in slaves_alive if self.dc_by_ip.get(ip) == self.current_dc]
            return by_dc if by_dc else slaves_alive


class DjangoRedisSentinelClient(DefaultClient):
    def __init__(self, server, params, backend):
        super(DjangoRedisSentinelClient, self).__init__(server, params, backend)

        self.sentinel_hosts = self._options['SENTINEL_HOSTS']
        self.sentinel_service_name = self._options['SENTINEL_SERVICE_NAME']
        self.current_dc = self._options['CURRENT_DC']

        sentinel_kwargs = {'socket_timeout': self._options['SENTINEL_SOCKET_TIMEOUT']}
        connection_kwargs = {
            'socket_timeout': self._options['SOCKET_TIMEOUT'],
            'password': self._options['PASSWORD']
        }

        self.sentinel = DjangoRedisSentinel(self.sentinel_hosts, sentinel_kwargs=sentinel_kwargs, current_dc=self.current_dc, **connection_kwargs)
        self.master = self.sentinel.master_for(self.sentinel_service_name)
        self.slave = self.sentinel.slave_for(self.sentinel_service_name)

    def call_method(self, method_name, *args, **kwargs):
        default = kwargs.pop('default_value', None)
        try:
            return getattr(super(DjangoRedisSentinelClient, self), method_name)(*args, **kwargs)
        except ConnectionInterrupted:
            log.exception('cache.{} failed'.format(method_name))
            return default
        except Exception:
            log.exception('unknown cache error. cache.{} failed'.format(method_name))
            return default

    def get(self, key, default=None, version=None, client=None):
        return self.call_method('get', key, default=default, version=version, client=client, default_value=default)

    def get_many(self, keys, version=None, client=None):
        return self.call_method('get_many', keys, version=version, client=client, default_value={})

    def set(self, key, value, timeout=DEFAULT_TIMEOUT, version=None, client=None, nx=False, xx=False):
        self.call_method('set', key, value, timeout=timeout, version=version, client=client, nx=nx, xx=xx)

    def set_many(self, data, timeout=DEFAULT_TIMEOUT, version=None, client=None):
        self.call_method('set_many', data, timeout=timeout, version=version, client=client)

    def delete(self, key, version=None, prefix=None, client=None):
        self.call_method('delete', key, version=version, prefix=prefix, client=client)

    def delete_many(self, keys, version=None, client=None):
        self.call_method('delete_many', keys, version=version, client=client)

    def add(self, key, value, timeout=DEFAULT_TIMEOUT, version=None, client=None):
        return self.call_method('set', key, value, timeout=timeout, version=version, client=client, nx=True, default_value=False)

    def get_client(self, write=True, tried=(), show_index=False):
        client = self.master if write else self.slave

        if show_index:
            # https://a.yandex-team.ru/arc/trunk/arcadia/contrib/python/django-redis/django_redis/client/default.py?rev=r4748001#L125
            # https://a.yandex-team.ru/arc/trunk/arcadia/contrib/python/django-redis/django_redis/client/default.py?rev=r4748001#L145
            # _slave_read_only = True
            return client, 0
        else:
            return client
