# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from threading import Lock

from django.conf import settings
from django.db import connections
from django.dispatch import Signal

from travel.rasp.library.python.common23.db import maintenance
from travel.rasp.library.python.common23.settings.utils import define_setting
from travel.rasp.library.python.common23.utils_db.threadutils import Ticker

from travel.rasp.library.python.db.replica_sync_checker import ReplicaSyncChecker


define_setting('DB_SWITCHER_BACKGROUND_SYNC_INTERVAL', default=10)


log = logging.getLogger(__name__)


class DbSwitcher(object):
    def __init__(self):
        self.roles = {
            role: 'STUB_DB' for role in settings.AVAILABLE_ROLES
        }
        self.cache_tag = None
        self.db_switched = Signal()  # изменились базы
        self.data_updated = Signal()  # измененились базы или CACHEROOT
        self.synced = False

    def sync_db(self, send_signal=True):
        """Обновляем настройки из maintenance."""
        old_db_alias = self.roles[settings.INSTANCE_ROLE.role]
        old_cache_tag = self.cache_tag

        self._update()

        new_db_alias = self.roles[settings.INSTANCE_ROLE.role]
        new_cache_tag = self.cache_tag
        self._update_cache_root(new_db_alias, new_cache_tag)

        if send_signal:
            if new_db_alias != old_db_alias:
                self.db_switched.send(self)

            if new_db_alias != old_db_alias or new_cache_tag != old_cache_tag:
                self.data_updated.send(self)

    @staticmethod
    def _close_connections(**kwargs):
        for conn in connections.all():
            conn.close()

    def sync_with_lazy_reconnect(self):
        self.db_switched.connect(self._close_connections)
        try:
            self.sync_db()
        finally:
            self.db_switched.disconnect(self._close_connections)

    def _update(self):
        try:
            conf = maintenance.read_conf()
        except Exception:
            # При любой проблеме с maintenance не падаем, а продолжаем использовать текущий конфиг
            # https://st.yandex-team.ru/RASPADMIN-1252
            log.exception(u'Unable to update maintenance config')
            if not self.synced or settings.MAINTENANCE_DB_CRITICAL:
                raise
        else:
            self.synced = True
            for db_role in self.roles.keys():
                self.roles[db_role] = conf[db_role]
            self.cache_tag = conf.get('cache_tag')

    def _update_cache_root(self, new_db_alias, new_cache_tag):
        settings.CACHEROOT = '/yandex/rasp/{}/{}/'.format(settings.PKG_VERSION, new_db_alias)

        if new_cache_tag:
            settings.CACHEROOT += '{}/'.format(new_cache_tag.replace(' ', '_'))

    def get_db_alias(self, db_role=None):
        if db_role is None:
            db_role = settings.INSTANCE_ROLE.role

        if db_role not in settings.AVAILABLE_ROLES:
            raise maintenance.UnknownRoleError(u'Unknown role: {}'.format(db_role))

        return self.roles[db_role]

    def swap(self, role1, role2):
        maintenance.swap(role1, role2)
        self.sync_db()

    def __repr__(self):
        return 'DBSwitcher: {}'.format(self.roles)


switcher = DbSwitcher()


# shortcuts
def work_alias():
    return switcher.get_db_alias(settings.WORK_DB)


def service_alias():
    return switcher.get_db_alias(settings.SERVICE_DB)


def instance_role_alias():
    return settings.INSTANCE_ROLE.role


def get_connection_by_role(role=None):
    alias = switcher.get_db_alias(role)

    dbwrapper = connections[alias]
    dbwrapper.ensure_connection()

    return dbwrapper.connection


def get_replica_sync_checker(alias, is_synced, *args, **kwargs):
    dbwrapper = connections[alias]
    return ReplicaSyncChecker(dbwrapper.get_all_hosts(), dbwrapper.get_connection_to_host, is_synced, *args, **kwargs)


class SyncDbInBackground(Ticker):
    """ Запускает sync_db в треде каждые interval секунд."""

    def __init__(self, interval=None, *args, **kwargs):
        interval = settings.DB_SWITCHER_BACKGROUND_SYNC_INTERVAL if interval is None else interval

        super(SyncDbInBackground, self).__init__(interval, *args, **kwargs)
        self.lock = Lock()

    def tick(self, *args, **kwargs):
        try:
            with self.lock:
                switcher.sync_db()
        except Exception:
            # Надеемся, что следующий тик не сломается, поэтому только логируем ошибку.
            log.exception('background sync_db failed')
