# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from contextlib import contextmanager
from datetime import timedelta

import six
from django.conf import settings
from pybreaker import CircuitBreaker

from travel.rasp.library.python.common23.settings.utils import define_setting
from travel.rasp.library.python.common23.date import environment
from travel.library.python.tracing.instrumentation import traced_function


DEFAULT_CACHE_TIME = 2 * 60

NOTHING = object()

log = logging.getLogger(__name__)

# динамические настройки используются часто и в самых неожиданных местах
# при этом в случае недоступности хранилища - в текущей логике мы возвращаем
# последнее закешированное значение => кажется что fail_max=1 это ок
define_setting('DYNAMIC_SETTINGS_BREAKER_PARAMS', default={'fail_max': 1, 'reset_timeout': 60})

dynamic_settings_breaker = CircuitBreaker(**settings.DYNAMIC_SETTINGS_BREAKER_PARAMS)


class DynamicSetting(object):
    def __init__(self, default_value, cache_time=DEFAULT_CACHE_TIME,
                 converter=None, name=None, storage=None, description='', required=True):
        """
        :param default_value: Значение по-умолчанию
        :param cache_time: Время кеша в секундах
        :param converter: Тип или функция конвертирования. По умолчанию это тип значения default_value
        :param name: Имя настройки, обычно передается в register_setting(s)
        :param storage: Хранилище, обычно назначается в register_setting(s)
        :param description: Описание
        :param required: Обязательно ли поле
        """
        self.name = name
        self.storage = storage
        self.default_value = default_value
        self.cached_at = None
        self.cached_value = NOTHING
        self.cache_time = timedelta(seconds=cache_time)
        self.description = description
        self.required = required

        if not converter:
            if type(default_value) in [int, float, bool, six.binary_type, six.text_type, list, dict]:
                converter = type(default_value)

        self.converter = converter

    @dynamic_settings_breaker
    @traced_function
    def get_from_storage(self):
        value = self.storage.get(self.name)
        return self.converter(self.default_value if value is NOTHING else value)

    def get(self):
        if self.cached_at is None or environment.now() - self.cached_at > self.cache_time:
            try:
                self.cached_value = self.get_from_storage()
            except Exception:
                log.exception('Ошибка при попытке обновить настройку %s', self.name)
                if self.cached_value is NOTHING:
                    self.cached_value = self.converter(self.default_value)
            self.cached_at = environment.now()

        return self.cached_value

    def set(self, value):
        value = self.converter(value)
        self.storage.set(self.name, value)
        self.cached_value = value


class Storage(object):
    def __init__(self):
        self._saving_context = {}  # for any Storage purposes

    def get(self, setting_name):
        """
        :return: value for a setting; must return dynamic_settings.NOTHING if value doesn't exist
        """
        raise NotImplementedError

    def set(self, setting_name, value):
        raise NotImplementedError

    def set_saving_context(self, **context):
        self._saving_context = context

    def get_setting_info(self, setting_name):
        return {'value': self.get(setting_name)}

    def setup(self):
        pass


class Settings(object):
    def __init__(self, storage, **settings):
        self._storage = storage
        self._settings = {}
        self.register_settings(**settings)

    def register_setting(self, setting_name, setting):
        if setting_name in self._settings:
            raise ValueError('Setting is already registered: {}'.format(setting_name))

        if not setting.name:
            setting.name = setting_name

        if not setting.storage:
            setting.storage = self._storage

        self._settings[setting_name] = setting

    def register_settings(self, **settings):
        for setting_name, setting in settings.items():
            self.register_setting(setting_name, setting)

    def get_settings(self):
        return dict(self._settings)

    def __setattr__(self, item, value):
        if item.startswith('_'):
            return super(Settings, self).__setattr__(item, value)

        if item not in self._settings:
            raise AttributeError('No such setting: {}'.format(item))

        self._settings[item].set(value)

    def __getattr__(self, setting_name):
        return self._settings[setting_name].get()

    @contextmanager
    def saving_context(self, **context):
        try:
            self._storage.set_saving_context(**context)
            yield
        finally:
            self._storage.set_saving_context()

    def get_setting_info(self, setting_name):
        stored_info = self._storage.get_setting_info(setting_name)
        setting = self._settings[setting_name]
        return {
            'default_value': setting.default_value,
            'cached_at': setting.cached_at,
            'cached_value': setting.cached_value,
            'cache_time': setting.cache_time,
            'description': setting.description,
            'stored_info': stored_info,
        }
