# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from builtins import str
from builtins import map
import os.path

import codecs
import inspect
import json
import logging
import six
import sys
import traceback
from collections import OrderedDict
from contextlib import contextmanager
from copy import copy
from datetime import datetime
from time import time
from ylog.context import get_log_context

from django.conf import settings
from django.core.files.utils import FileProxyMixin
from django.utils.encoding import force_bytes, force_text
from travel.rasp.library.python.common23.utils.exceptions import SimpleUnicodeException
from travel.rasp.library.python.common23.utils.files.fileutils import get_my_caller_file

STDOUT_DEFALUT_FD = 1
STDERR_DEFALUT_FD = 2

# !!!! Logging hack
logging._srcfile = os.path.normcase(logging.log.__code__.co_filename)


def patchedHandleError(self, record):
    """
    Handle errors which occur during an emit() call.

    This method should be called from handlers when an exception is
    encountered during an emit() call. If raiseExceptions is false,
    exceptions get silently ignored. This is what is mostly wanted
    for a logging system - most users will not care about errors in
    the logging system, they are more interested in application errors.
    You could, however, replace this with a custom handler if you wish.
    The record which was being processed is passed in to this method.
    """
    if logging.raiseExceptions and sys.stderr:  # see issue 13807
        ei = sys.exc_info()
        try:
            traceback.print_exception(ei[0], ei[1], ei[2],
                                      None, sys.stderr)
            traceback.print_stack(None, None, sys.stderr)
            sys.stderr.write('%s Logged from file %s, line %s\n' % (
                             datetime.now(), record.filename, record.lineno))
        except IOError:
            pass    # see issue 5971
        finally:
            del ei


logging.Handler.handleError = patchedHandleError


class LogError(SimpleUnicodeException):
    pass


class WatchedFileHandler(logging.FileHandler):
    def _open(self):
        self.encoding = self.encoding or 'utf-8'

        return WatchedFile(self.baseFilename, mode=self.mode, encoding=self.encoding)


class WatchedFile(FileProxyMixin):
    """Файл, которые переоткрывается при его переименовании или удалении"""

    def __init__(self, filename, mode='a', encoding='utf-8'):
        self.file = None
        self.mode = mode
        self._encoding = encoding

        self.filename = filename
        self.dirname = os.path.dirname(self.filename)

        if not os.path.exists(self.filename):
            self.dev, self.ino = None, None
        else:
            stat = os.stat(self.filename)
            self.dev, self.ino = stat.st_dev, stat.st_ino

        self.open()

    def open(self):
        if not os.path.exists(self.dirname):
            os.makedirs(self.dirname)

        self.close()

        self.file = codecs.open(self.filename, self.mode, encoding=self._encoding)

        st = os.stat(self.filename)
        self.dev, self.ino = st.st_dev, st.st_ino

    def reopen(self):
        self.open()

    def flush(self):
        self.file and self.file.flush()

    def write(self, msg):
        if not os.path.exists(self.filename):
            changed = True

        else:
            st = os.stat(self.filename)
            the_same = (st.st_dev == self.dev) and (st.st_ino == self.ino)

            changed = not the_same

        if changed:
            self.reopen()

        if isinstance(msg, six.text_type):
            self.file.write(msg)
        else:
            self.file.write(force_text(msg, encoding='utf-8', errors='replace'))

    def close(self):
        if not self.file:
            return

        if not self.file.closed:
            self.file.flush()
            self.file.close()


class FileTreeHandler(logging.Handler):
    """
    Обработчик, создающий дерево из обрабатываемых логов
    """

    SUFFIX = '.log'

    def __init__(self, base_path):
        """
        Initialize the handler.
        """
        logging.Handler.__init__(self)
        self.base_path = base_path
        self.streams = {}

    def get_stream(self, name):
        if name in self.streams:
            return self.streams[name]

        parts = name.split('.')

        path = os.path.join(self.base_path, *parts) + self.SUFFIX

        stream = self.streams[name] = WatchedFile(path)

        return stream

    def flush(self):
        """
        Flushes the streams.
        """

        for stream in self.streams.values():
            stream.flush()

    def emit(self, record):
        """
        Emit a record.

        If a formatter is specified, it is used to format the record.
        The record is then written to the stream with a trailing newline.
        """

        try:
            msg = self.format(record)

            stream = self.get_stream(record.name)

            stream.write("%s\n" % msg)

            stream.flush()

        except (KeyboardInterrupt, SystemExit):
            raise
        except:
            self.handleError(record)


class ExcludeFromTree(logging.Filter):
    """
    Исключает из корневого лога дочерние логи.

    >>> logging_config = {
    >>>     'filters': {
    >>>         'exclude_from_tree': {
    >>>             '()': 'travel.rasp.library.python.common23.logging.ExcludeFromTree',
    >>>             'starts_with_names': ('special.',),
    >>>             'exact_names': ('__main__',),
    >>>             'ignore_names': ('special.pypy',)
    >>>         }
    >>>     }
    >>>     'handlers': {
    >>>         'tree_handler': {
    >>>             'level': 'DEBUG',
    >>>             'class': 'travel.rasp.library.python.common23.logging.FileTreeHandler',
    >>>             'base_path': os.path.join(log_path, 'tree'),
    >>>             'formatter': 'verbose',
    >>>             'filters': ['exclude_from_tree']
    >>>         }
    >>>     }
    >>> }

    Не будут логироваться все логи начинающиеся со special. и __main__.
    special.pypy при это продолжит логироваться.
    """
    def __init__(self, exact_names=tuple(), starts_with_names=tuple(), ignore_names=tuple()):
        self.exact_names = exact_names
        self.starts_with_names = starts_with_names
        self.ignore_names = ignore_names

    def filter(self, record):
        if record.name in self.exact_names:
            return False

        if record.name in self.ignore_names:
            return True

        if record.name.startswith(self.starts_with_names):
            return False

        return True


@contextmanager
def log_run_time(title=u'', print_before=True, logger=None, log_level=logging.INFO):
    def prnt(*args):
        msg = u' '.join(map(six.text_type, args))
        if logger:
            logger.log(log_level, u' '.join(map(six.text_type, args)))
        else:
            print(msg)

    start = time()
    if print_before:
        prnt(u'{} -> started'.format(title))

    run_time = None

    def time_getter():
        return run_time

    try:
        yield time_getter
    finally:
        run_time = time() - start
        prnt(u'{} -> done in {:.4}'.format(title, run_time))


def get_file_logger_name(py_file):
    """
    Получаем название лога для файла проекта
    """
    if not py_file.endswith(u'.py'):
        raise LogError(u"%s не является файлом проекта не смогли создать лог" % py_file)

    file_path = os.path.abspath(py_file)
    project_path = os.path.abspath(settings.PROJECT_PATH)

    relative_path = file_path.replace(project_path, u'', 1).lstrip(os.path.sep)
    logger_name = relative_path.replace(os.path.sep, u'.')
    logger_name = u'.'.join(logger_name.split(u'.')[:-1])  # remove .py(pyc) extension
    parts = logger_name.split(u'.')
    if parts[-1] == '__init__':
        del parts[-1]
        logger_name = u'.'.join(parts)

    return logger_name


def capture_stdstreams_to_file(file):
    """
    Перенаправляем стандартные потоки в этот файл
    """
    fd = file.fileno()

    sys.stdout.flush()
    os.close(STDOUT_DEFALUT_FD)
    os.dup2(fd, STDOUT_DEFALUT_FD)

    sys.stderr.flush()
    os.close(STDERR_DEFALUT_FD)
    os.dup2(fd, STDERR_DEFALUT_FD)


class ContextRaspFormatter(logging.Formatter):
    def __init__(self, fmt=None, datefmt=None):
        if fmt:
            self._simple_fmt = fmt
        else:
            self._simple_fmt = "%(message)s"

        if ':' in self._simple_fmt:
            self._context_fmt = '%s: %%(context)s:%s' % tuple(self._simple_fmt.split(':', 1))
        else:
            self._context_fmt = self._simple_fmt + ' : ' + '%(context)s'

        self.datefmt = datefmt

    @classmethod
    def ensure_text(cls, text):
        if not isinstance(text, six.text_type):
            return force_text(text, encoding='utf-8', errors='replace')
        else:
            return text

    def formatException(self, ei):
        """
        Новая реализация
        """
        exc_info = traceback.format_exception(ei[0], ei[1], ei[2])
        exc_info = u"".join([ContextRaspFormatter.ensure_text(text) for text in exc_info]).strip()
        return exc_info

    def format(self, record):
        if hasattr(record, 'context') and record.context:
            self._fmt = self._context_fmt
        else:
            self._fmt = self._simple_fmt

        if record.name == '__main__' and getattr(record, 'main_file_path', None):
            record = copy(record)
            record.name = record.main_file_path

        return logging.Formatter.format(self, record)


def _default_tail_name():
    from travel.rasp.library.python.common23.date.environment import now
    return now().strftime('%Y-%m-%d_%H%M%S.log')


def create_run_log(log_name, format=None, capture_stdstreams=False, tail_name=_default_tail_name):
    from django.conf import settings
    format = format or settings.LOG_FORMAT

    path = os.path.join(settings.LOG_PATH, *log_name.split('.'))
    path = os.path.join(path, tail_name())

    if not os.path.exists(os.path.dirname(path)):
        os.makedirs(os.path.dirname(path))

    handler = logging.FileHandler(path)
    handler.setFormatter(ContextRaspFormatter(format))

    log = logging.getLogger()
    log.addHandler(handler)

    if capture_stdstreams:
        capture_stdstreams_to_file(handler.stream)

    return handler.stream


def create_current_file_run_log(format=None, capture_stdstreams=False, tail_name=_default_tail_name):
    stack = inspect.stack()
    caller_file = get_my_caller_file(stack)

    log_name = 'special.script_runs.' + get_file_logger_name(caller_file)

    return create_run_log(log_name, format, capture_stdstreams=capture_stdstreams, tail_name=tail_name)


class AddContextFilter(logging.Filter):
    """
    Добавляет в LogRecord аттрибут `context`, содержащий все данные из текущего pylogctx контекста.
    Для описания того, как прокинуть данные в контекст - см. https://github.com/novafloss/pylogctx

    P.S.: Для того, чтобы данные попали в лог - нужно модифицировать строку формата,
     например так: `_format = '%(levelname)s %(asctime)s %(process)d %(context)s %(name)s %(message)s'`.
    """
    def __init__(self, as_dict=False, default=None, name='', context_field='context'):
        super(AddContextFilter, self).__init__(name)
        self.as_dict = as_dict
        self.default = default or {}
        self.context_field = context_field

    def filter(self, record):
        context_data = OrderedDict(get_log_context().items())
        for k, v in self.default.items():
            if k not in context_data:
                context_data[k] = v

        if not self.as_dict:
            context_data = ', '.join('{}={}'.format(k, v) for k, v in context_data.items())
        setattr(record, self.context_field, context_data)
        return True


class FieldsInContextFilter(logging.Filter):
    """
    Фильтр, для логирования записей, содержащий required_fields в контексте.
    """
    def __init__(self, required_fields, name=''):
        super(FieldsInContextFilter, self).__init__(name)
        self.required_fields = required_fields

    def filter(self, record):
        return all(f in get_log_context() for f in self.required_fields)


class RequiredFieldsFilter(logging.Filter):
    """
    Фильтр для логирования записей, содержащих required_fields.
    """
    def __init__(self, required_fields, name=''):
        super(RequiredFieldsFilter, self).__init__(name)
        self.required_fields = required_fields

    def filter(self, record):
        return all(val for key, val in record.__dict__.items() if key in self.required_fields)


class _DictFormatter(logging.Formatter):
    DEFAULT_KEYS_TO_LOG = {
        'levelname', 'asctime', 'process', 'context', 'name'
    }

    def __init__(self, keys_to_log=None, omit_mandatory=None, *args, **kwargs):
        super(_DictFormatter, self).__init__(*args, **kwargs)
        self.keys_to_log = keys_to_log or self.DEFAULT_KEYS_TO_LOG
        self.omit_mandatory = omit_mandatory

    def log_record_to_dict(self, record):
        record_dict = {k: v for k, v in record.__dict__.items() if k in self.keys_to_log}
        if not self.omit_mandatory:
            record_dict['message'] = record.getMessage()
            record_dict['created'] = self.formatTime(record, self.datefmt)
            record_dict['created_utc'] = datetime.utcfromtimestamp(record.created).isoformat() + 'Z'
        # Decorate with exception info
        if record.exc_info is not None:
            record_dict.update({
                'exception': {
                    'message': str(record.exc_info[1]),
                    'traceback': self.formatException(record.exc_info)
                }
            })
        return record_dict

    def format(self, record):
        raise NotImplementedError


class JsonFormatter(_DictFormatter):
    def format(self, record):
        record_dict = super(JsonFormatter, self).log_record_to_dict(record)
        if 'context' in record_dict and isinstance(record_dict['context'], dict):
            record_dict['context'] = {
                k: force_text(v) for k, v in record_dict['context'].items()
            }

        try:
            return force_bytes(json.dumps(record_dict, ensure_ascii=False))
        except TypeError:
            return force_bytes(json.dumps(
                {'created': record_dict['created'], 'message': record_dict['message']},
                ensure_ascii=False))


class WarningFilterOnce(AddContextFilter):
    def __init__(self):
        super(WarningFilterOnce, self).__init__()
        self._logged = set()

    def filter(self, record):
        if not super(WarningFilterOnce, self).filter(record):
            return False

        try:
            hash(record.args)
        except TypeError:
            pass
        else:
            if record.args in self._logged:
                return False
            else:
                self._logged.add(record.args)

        return True
