# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
from collections import OrderedDict

from django.utils.encoding import force_text
from ylog.format import format_post


class BaseJsonFormatter(logging.Formatter):
    BASE_FIELDS = ('name', 'funcName', 'process')

    def __init__(self, fmt=None, datefmt=None, extra_base_fields=None, tag=None, extra_context=None, add_request_info=False):
        super(BaseJsonFormatter, self).__init__(fmt=fmt, datefmt=datefmt)

        self.tag = tag
        self.fields = self.BASE_FIELDS + (extra_base_fields or ())
        self.extra_context = extra_context
        self.add_request_info = add_request_info

    def make_data(self, record):
        raise NotImplementedError

    def make_std_fields(self, record):
        raise NotImplementedError

    def format(self, record):
        data = self.make_data(record)
        fields = {'std': self.make_std_fields(record)}

        if self.tag:
            fields['tag'] = self.tag

        self._add_context(record, fields)

        data['@fields'] = fields

        return json.dumps(data)

    def _get_context(self, record):
        context = OrderedDict()
        if self.extra_context:
            context.update(self.extra_context)
        if hasattr(record, 'context') and record.context:
            try:
                context.update(record.context)
            except (TypeError, ValueError):
                context['bad_context'] = True
                context['context_value'] = repr(record.context)

        if hasattr(record, 'request') and self.add_request_info:
            request_context = OrderedDict([
                ('path', record.request.path),
                ('method', record.request.method),
                ('query', force_text(record.request.GET)),
                ('post', force_text(format_post(record.request)))
            ])
            request_context.update(context)
            return request_context
        else:
            return context

    def _add_context(self, record, fields):
        context = self._get_context(record)

        if context:
            fields['context'] = context
