# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
from collections import OrderedDict

from django.utils.encoding import force_text
from ylog.format import format_post


class QloudJsonFormatter(logging.Formatter):
    """
    Копипаста с ylog, но с нашим контекстом
    https://docs.qloud.yandex-team.ru/doc/logs#json

    Форматирует сообщение как JSON в формате Qloud.

    Сообщение будет иметь следующий формат:
    {
        "msg": "message",
        "stackTrace": "your very long stacktrace \n with newlines \n and all the things",
        "level": "WARN",
        "@fields": {
            "std": {
                "funcName": "get_values",
                "lineno": 274,
                "name": "mymicroservice.dao.utils",
            },
            "context": {
                "foo": "qwerty",
                "bar": 42,
            }
        }
    }

    В словарь @fields.context попадут поля, перечисленные в log_context.
    """

    BASE_FIELDS = ('name', 'funcName', 'process')

    def __init__(self, fmt=None, datefmt=None, extra_base_fields=None, tag=None,
                 extra_context=None):
        super(QloudJsonFormatter, self).__init__(fmt=fmt, datefmt=datefmt)

        self.tag = tag
        self.fields = self.BASE_FIELDS + (extra_base_fields or ())
        self.extra_context = extra_context

    def _get_message(self, record):
        return record.getMessage()

    def format(self, record):
        log_data = {
            'message': self._get_message(record),
            'level': record.levelname,
        }

        if record.exc_info:
            log_data['stackTrace'] = self.formatException(record.exc_info)

        fields = {}
        if self.tag:
            fields['tag'] = self.tag

        standard_fields = self._get_standard_fields(record)
        if standard_fields:
            fields['std'] = standard_fields

        self._add_context(record, fields)

        if fields:
            log_data['@fields'] = fields

        return json.dumps(log_data)

    def _get_context(self, record):
        context = OrderedDict()
        if self.extra_context:
            context.update(self.extra_context)
        if hasattr(record, 'context') and record.context:
            try:
                context.update(record.context)
            except (TypeError, ValueError):
                context['bad_context'] = True
                context['context_value'] = repr(record.context)

        return context

    def _add_context(self, record, fields):
        context = self._get_context(record)

        if context:
            fields['context'] = context

    def _get_standard_fields(self, record):
        return OrderedDict(
            (field, getattr(record, field))
            for field in self.fields
            if hasattr(record, field)
        )


class QloudExceptionFormatter(QloudJsonFormatter):
    def __init__(self, *args, **kwargs):
        self.add_request_info = kwargs.pop('add_request_info', False)
        super(QloudExceptionFormatter, self).__init__(*args, **kwargs)

    def _get_context(self, record):
        super_context = super(QloudExceptionFormatter, self)._get_context(record)
        if hasattr(record, 'request') and self.add_request_info:
            context = OrderedDict([
                ('path', record.request.path),
                ('method', record.request.method),
                ('query', force_text(record.request.GET)),
                ('post', force_text(format_post(record.request)))
            ])
            context.update(super_context)
            return context
        else:
            return super_context

    def formatException(self, ei):
        return super(QloudJsonFormatter, self).formatException(ei)


def context_extractor(request):
    return {
        'user_rid': request.GET.get('_rid', ''),
        'service_rid': request.META.get('HTTP_X_REQUEST_ID', '')
    }
