# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.db import models
from django.utils.translation import ugettext_lazy as _

from travel.rasp.library.python.common23.models.core.directions.schema import parse_schema, build_schema, initial_schema
from travel.rasp.library.python.common23.models.core.geo.const import RE_MDOT
from travel.rasp.library.python.common23.models.core.geo.station import Station
from travel.rasp.library.python.common23.models.fields import TextFileField
from travel.rasp.library.python.common23.models.precache.manager import PrecachingManager
from travel.rasp.library.python.common23.models.texts.i18n import L_field


class ExternalDirection(models.Model):
    u"""Направления для электричек для показа в городах"""
    L_title = L_field(_(u'название направления'), add_local_field=True, local_field_critical=True, max_length=100)

    L_full_title = L_field(_(u'полное название'), add_local_field=True, local_field_critical=True, max_length=100)

    code = models.CharField(verbose_name=_(u'код направления'),
                            max_length=50, blank=False, null=False, unique=True)
    suburban_zone = models.ForeignKey("SuburbanZone", blank=True, null=True,
                                      verbose_name=_(u"пригородная зона"),
                                      help_text=_(u"пригородная зона, в которую входит направление"))
    base_station = models.ForeignKey(Station, blank=True, null=True,
                                     verbose_name=_(u"базовый вокзал"))

    reduced_tariff = models.FloatField(_(u'множитель льготного тарифа'), default=0)

    use_synchronous_tariff = models.BooleanField(_(u'использовать синхронный тариф'), null=False, default=False)

    objects = PrecachingManager(keys=['pk', 'code'])

    def L_cutted_full_title(self, lang=None):
        # RASP-11929
        # TODO адаптировать для переводов
        # if not lang:
        #     lang = translation.get_language()

        return self.L_title().replace(u'направление', u'напр.')

    @property
    def schema_file_name(self):
        return "%s.csv" % self.code

    schema_file = TextFileField(blank=True, verbose_name=_(u"файл с картой направления"),
                                encoding='cp1251',
                                help_text=_(u"принимаются только текстовые файлы в кодировке cp1251"))

    @property
    def title_gen(self):
        return self.title[:-2] + u'ого'

    @property
    def title_dat(self):
        return self.title[:-2] + u'ому'

    @property
    def stations(self):
        return Station.objects.filter(hidden=False,
                                      externaldirectionmarker__external_direction=self).order_by('title')

    def __unicode__(self):
        return "%s (%s) " % (self.title, self.code)

    def schema(self, edit=False):
        if not self.schema_file:
            if edit:
                return initial_schema(self)

            return None

        zones = parse_schema(self.schema_file, self.code, edit)
        # Заполним автобусные остановки моделями
        bus_sids = [int(sid)
                    for num, rows in zones
                    for row in rows
                    for item, href, label, metro, busstations in row if busstations
                    for sid in busstations]

        bus_mapping = Station.hidden_manager.filter(pk__in=bus_sids).all()
        bus_mapping = dict([(s.id, s) for s in bus_mapping])

        for num, rows in zones:
            for row in rows:
                for i, item in enumerate(row):
                    item, href, label, metro, busstations = item

                    if busstations:
                        busstations = [bus_mapping[sid] for sid in busstations if sid in bus_mapping]
                        busstations = [(s.get_popular_title(), s.get_schedule_url('bus')) for s in busstations]
                        busstations.sort(key=lambda x: RE_MDOT.sub('', x[0]).lower())

                        row[i] = item, href, label, metro, busstations

        return zones

    def parse_schema_json(self, json):
        self.schema_file = build_schema(json)

    @classmethod
    def find_for_stations(cls, *stations):
        """id направления для данных станций"""

        directions = None

        for s in stations:
            ds = set(m.external_direction_id for m in s.externaldirectionmarker_set.all())

            if directions is None:
                directions = ds
            else:
                directions.intersection_update(ds)

                # Если сет уже пустой, то дальше
                # он не пополнится
                if not directions:
                    return []

        return directions

    class Meta(object):
        ordering = ('title',)
        verbose_name = _(u'внешнее направление')
        verbose_name_plural = _(u'внешние направления')
        app_label = 'www'
        db_table = 'www_externaldirection'
