# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from future import standard_library
standard_library.install_aliases()
from builtins import str
import random
import re
import six
from lxml import etree

from django.conf import settings
from django.contrib.contenttypes.fields import GenericRelation
from django.db import models
from django.utils import translation
from django.utils.encoding import force_text, smart_str
from django.utils.translation import ugettext_lazy as _, get_language

from travel.rasp.library.python.common23.db.mds.s3_storage import mds_s3_media_storage
from travel.rasp.library.python.common23.models.core.geo.const import (  # noqa
    RE_MDOT, TR_SUGGEST_FULL_TITLE_OVERRIDES, NOTHERN_CYPRUS_CITIES
)
from travel.rasp.library.python.common23.models.core.geo.point_slugs import PointSlugField, make_station_slug, POINT_SLUG_LENGTH
from travel.rasp.library.python.common23.models.core.geo.station_code_manager import StationCodeManager
from travel.rasp.library.python.common23.models.core.geo.timezone_mixin import TimeZoneMixin
from travel.rasp.library.python.common23.models.core.geo.translocal_mixin import TransLocalMixin
from travel.rasp.library.python.common23.models.core.geo.point import Point
from travel.rasp.library.python.common23.models.core.geo.station_type import StationType
from travel.rasp.library.python.common23.models.fields import CodeCharField, TrimmedCharField
from travel.rasp.library.python.common23.models.precache.manager import PrecachingManager
from travel.rasp.library.python.common23.models.texts.i18n import L_field
from travel.rasp.library.python.common23.models.utils import HiddenManagerWrapper

from travel.rasp.library.python.common23.utils.caching import cache_method_result
from travel.rasp.library.python.common23.utils.text import nbsp, cut_direction
from travel.rasp.library.python.common23.utils_db.locations import get_params, composeurl
from travel.rasp.library.python.common23.xgettext.i18n import gettext, xgettext


TABLO_STATE_CHOICES = (
    ('', _(u'нет табло')),
    ('statuses', _(u'только статусы')),
    ('real', _(u'статусы и время')),
    ('nodata', _(u'нет данных'))
)


USE_DIRECTION_CHOICES = (
    ('', _(u'не использует направления')),
    ('dir', _(u'использует направления')),
    ('subdir', _(u'использует поднаправления'))
)


AGREED_STOP_TYPE_CHOICES = (
    ('', _('неизвестно')),
    ('agreed', _('согласованная остановка')),
    ('not_agreed', _('несогласованная остановка')),
)


class BaseStation(models.Model, TimeZoneMixin, TransLocalMixin, Point):
    u"""Станция"""
    ADLER_ID = 9613054

    L_address = L_field(_(u'адрес'), add_local_field=True,
                        max_length=200, null=True, blank=True, default=None)

    majority = models.ForeignKey('www.StationMajority',
                                 verbose_name=_(u'важность станции'),
                                 null=False, blank=False)

    settlement = models.ForeignKey('www.Settlement', verbose_name=_(u'город'),
                                   null=True, blank=True, default=None)
    district = models.ForeignKey('www.District', null=True, blank=True, default=None,
                                 verbose_name=_(u"район"))
    region = models.ForeignKey('www.Region', verbose_name=_(u'область'),
                               null=True, blank=True, default=None)
    country = models.ForeignKey('www.Country', verbose_name=_(u'страна'),
                                null=True, blank=True, default=None)

    # Переводы
    L_title = L_field(
        _(u'наименование'), add_local_field=True, local_field_critical=True, add_override_field=True, db_index=True,
        extra={
            'ru': [
                ('preposition_v_vo_na', L_field.char_field(
                    _(u'предлог (в, во, на)'), max_length=2,
                    help_text=_(u'в, во, на – направление: куда? (во Владимир, на ул. Пушкина)'),
                )),
                ('genitive', L_field.char_field(
                    _(u"наименование (род. падеж)"),
                    help_text=_(
                        u'откуда? - от (кого? чего?) Москвы, из (кого? чего?) Москвы, табло (кого? чего?) Москвы')
                )),
                ('accusative', L_field.char_field(
                    _(u'наименование (вин. падеж)'),
                    help_text=_(u'куда? - в (кого? что?) Москву')
                )),
                ('locative', L_field.char_field(
                    _(u"наименование (пред. падеж)"),
                    help_text=_(u'где? - в (ком? чем?) Москве')
                )),
            ]
        }
    )

    L_popular_title = L_field(
        _(u'народное наименование'), add_local_field=True, db_index=True,
        extra={
            'ru': [
                ('genitive', L_field.char_field(
                    _(u'народное наименование (род. падеж)'), db_index=True,
                    help_text=_(
                        u'откуда? - от (кого? чего?) Москвы, из (кого? чего?) Москвы, табло (кого? чего?) Москвы')
                )),
            ]
        },
        fallback_fields=['L_title'],
        help_text=_(u'популярное в народных массах название'),
    )

    L_short_title = L_field(
        _(u'короткое наименование'), add_local_field=True,
        fallback_fields=['L_title'],
    )

    slug = PointSlugField(verbose_name=_(u'название для ссылок'), max_length=POINT_SLUG_LENGTH,
                          null=True, unique=True, blank=True)

    t_type = models.ForeignKey('www.TransportType', verbose_name=_(u'тип транспорта'),
                               default=1)
    type_choices = TrimmedCharField(_(u'типы отображения табло/расписания'), max_length=100,
                                    null=True, default=u'', blank=True)
    hidden = models.BooleanField(help_text=_(u'не показывать нигде, кроме рейсов следования. '
                                             u'Не давать ссылку в рейсе следования.'),
                                 verbose_name=_(u"скрыта"),
                                 default=False)
    express_id = CodeCharField(max_length=50, null=True,
                               blank=True, default=None, editable=False,
                               unique=True)
    sirena_id = CodeCharField(verbose_name=_(u'код в "Sirena"'),
                              max_length=10, null=True,
                              blank=True, default=None, editable=False,
                              unique=True)
    time_zone = models.CharField(verbose_name=_(u'временная зона'),
                                 help_text=_(u'временная зона в международном формате'),
                                 null=False, blank=True, max_length=30)
    time_zone_not_check = models.BooleanField(verbose_name=_(u"не сверять временную зону с родительским регионом"),
                                              default=False)
    site_url = models.URLField(verbose_name=_(u'урл сайта'), max_length=100,
                               null=True, blank=True, default=None)
    # FIXME: возможно не используемое поле, стоит его удалить
    properties = models.ManyToManyField('www.StationProperty', verbose_name=_(u'свойства'), blank=True, default=None)
    # FIXME: возможно не используемое поле, стоит его удалить
    _majority = models.IntegerField(verbose_name=_(u'важность'),
                                    null=True, blank=True, default=None)

    L_how_to_get_to_city = L_field(_(u'как доехать до города'), null=True, blank=True, field_cls=models.TextField)

    not_generalize = models.BooleanField(
        verbose_name=_(u'не обобщать до города при именовании рейсов'),
        blank=True, default=False
    )

    # RASP-1546
    station_type = models.ForeignKey(StationType, verbose_name=_(u'тип станции'), default=12)
    longitude = models.FloatField(verbose_name=_(u"долгота"), default=None,
                                  blank=True, null=True, db_index=True)
    latitude = models.FloatField(verbose_name=_(u"широта"), default=None,
                                 blank=True, null=True, db_index=True)

    # RASP-3160
    map_zoom = models.IntegerField(verbose_name=_(u'масштаб карты'), default=None,
                                   blank=True, null=True)

    use_direction = models.CharField(verbose_name=_(u"связь с направлениями"),
                                     blank=True, default='subdir',
                                     choices=USE_DIRECTION_CHOICES,
                                     max_length=10)
    suburban_zone = models.ForeignKey("SuburbanZone", verbose_name=_(u'пригородная зона'),
                                      null=True, blank=True, default=None)

    has_aeroexpress = models.BooleanField(_(u'по станции ходят аэроэкспрессы'),
                                          blank=True, null=False, default=False)

    near_metro = models.CharField(verbose_name=_(u'рядом с метро'), default=None,
                                  blank=True, null=True, max_length=25)
    last_change = models.DateTimeField(verbose_name=_(u'дата последней проверки'), default=None,
                                       blank=True, null=True,
                                       help_text=_(u"выставляется при ручной проверке адреса, названий и тд."))

    # RASP-3855
    photo = models.ImageField(verbose_name=_(u'фото станции'),
                              storage=mds_s3_media_storage,
                              upload_to='data/station/photo',
                              null=True, blank=True, default=None)
    schema_image = models.ImageField(verbose_name=_(u'Схема станции'),
                                     storage=mds_s3_media_storage,
                                     upload_to='data/station/schema_image',
                                     null=True, blank=True, default=None)

    panorama_url = models.URLField(verbose_name=_(u"ссылка на панораму"), null=True, blank=True)

    # RASP-4135
    show_settlement = models.BooleanField(verbose_name=_(u"показывать город в адресе"), default=True,
                                          help_text=_(u"писать ли перед адресом название города станции"))

    tablo_state = models.CharField(choices=TABLO_STATE_CHOICES, default="",
                                   verbose_name=_(u"табло"), max_length=10,
                                   blank=True)

    tablo_state_prev = models.CharField(choices=TABLO_STATE_CHOICES, default="",
                                        verbose_name=_(u"табло (предыдущее значение)"), max_length=10,
                                        blank=True)

    supplier = models.ForeignKey('www.Supplier', verbose_name=_(u'поставщик'), default=None, blank=True, null=True)

    synonyms = GenericRelation('PointSynonym')

    fuzzy_only = models.BooleanField(_(u"только нечеткие rtstations"), default=False)
    virtual_end = models.BooleanField(_(u"виртуальная конечная"), default=False)

    incomplete_bus_schedule = models.BooleanField(verbose_name=_(u'неполное расписание автобусов'),
                                                  default=False)

    SHOW_MODE_CHOICES = (
        ('block', _(u'блочное расписание')),
        ('simpl', _(u'построчное расписание')),
    )
    show_mode = models.CharField(verbose_name=_(u"режим отображения автобусов"),
                                 blank=False, default='block', choices=SHOW_MODE_CHOICES,
                                 max_length=5)

    meta_title = models.CharField(max_length=255, verbose_name=_(u'meta title'), null=True, blank=True)
    meta_description = models.TextField(verbose_name=_(u'meta description'), null=True, blank=True)

    # fuzzy sign group
    is_fuzzy = models.BooleanField(_(u'нечеткое время отправления/прибытия'), default=False)
    is_searchable_to = models.BooleanField(_(u'искать до станции'), default=True)
    is_searchable_from = models.BooleanField(_(u'искать от станции'), default=True)
    in_station_schedule = models.BooleanField(_(u'в расписании по станции'), default=True)
    in_thread = models.BooleanField(_(u'показывать на странице нитки'), default=True)

    show_tablo_stat = models.BooleanField(_(u"показывать отмены и задержки по станции на главной странице Расписаний"),
                                          default=True)
    modified_at = models.DateTimeField(_(u'Дата-время изменения'), auto_now=True, null=False, blank=True)

    use_in_departure_forecast = models.BooleanField(_(u'использовать при прогнозировании опозданий для поездов, '
                                                      u'которые не ушли cо станции отправления'), default=True)
    use_in_forecast = models.BooleanField(_(u'использовать при прогнозировании опозданий для поездов, '
                                            u'которые не пришли на следующую станцию'), default=False)

    agreed_stop_type = models.CharField(verbose_name=_(u'согласованная остановка'),
                                        max_length=15, choices=AGREED_STOP_TYPE_CHOICES,
                                        default='', null=False, blank=True)

    objects = PrecachingManager(keys=['pk'])
    hidden_manager = HiddenManagerWrapper('objects')
    code_manager = StationCodeManager()

    class Meta(object):
        abstract = True
        verbose_name = _(u'станция')
        verbose_name_plural = _(u'станции')
        app_label = 'www'
        db_table = 'www_station'

    @classmethod
    def fill_station(cls, data):
        s = cls()
        for key, value in six.iteritems(data):
            setattr(s, key, value)

        return s

    @property
    def is_block_schedule(self):
        return self.show_mode == 'block'

    @property
    def type_choices_set(self):
        return self.type_choices and set(self.type_choices.split(',')) or set()

    @property
    def page_type(self):
        '''
        Тип страницы станции
        '''
        return self._get_page_info()[0]

    @property
    def subtypes(self):
        '''
        Cписок подтипов страницы станции (в основном по типу транспорта)
        '''
        return self._get_page_info()[1]

    @property
    def main_subtype(self):
        '''
        Подтип страницы, выбираемый по умолчанию
        '''
        return self._get_page_info()[2]

    @cache_method_result
    def _get_page_info(self):
        from travel.rasp.library.python.common23.models.transport.transport_type import TransportType

        subtypes = []
        main_subtype = None
        page_type = None

        if self.t_type.id in {TransportType.TRAIN_ID, TransportType.SUBURBAN_ID}:
            if 'train' in self.type_choices_set:
                subtypes.append('train')
            if 'suburban' in self.type_choices_set:
                subtypes.append('suburban')
            if 'train' in self.type_choices_set and 'suburban' in self.type_choices_set:
                subtypes.append('tablo')

            if 'train' in subtypes:
                main_subtype = 'train'
            elif 'suburban' in subtypes:
                main_subtype = 'suburban'

            page_type = 'train'

        elif self.t_type.id == TransportType.PLANE_ID:
            if 'tablo' in self.type_choices_set:
                subtypes.append('plane')
                main_subtype = 'plane'

            page_type = 'plane'

        else:
            if 'schedule' in self.type_choices_set:
                subtypes.append('schedule')
                main_subtype = 'schedule'

            page_type = (
                'bus'
                if self.t_type.id in {TransportType.BUS_ID, TransportType.PSEUDO_GORTRANS}
                else 'water'
            )

        return page_type, subtypes, main_subtype

    def get_externaldirections(self):
        from travel.rasp.library.python.common23.models.core.directions.external_direction import ExternalDirection
        return ExternalDirection.objects.filter(externaldirectionmarker__station=self).distinct()

    def get_code(self, system):
        from travel.rasp.library.python.common23.models.core.geo.code_system import CodeSystem
        from travel.rasp.library.python.common23.models.core.geo.station import Station
        from travel.rasp.library.python.common23.models.core.geo.station_code import StationCode

        if isinstance(system, CodeSystem):
            system_code = system.code
        else:
            system_code = system

        if Station.code_manager.code_cache:
            code = Station.code_manager.getcodes(self).get(system_code)
            if code:
                return code
        try:
            return self.code_set.get(system__code=system_code).code
        except StationCode.DoesNotExist:
            return None

    def set_code(self, system_code, code):
        from travel.rasp.library.python.common23.models.core.geo.code_system import CodeSystem
        from travel.rasp.library.python.common23.models.core.geo.station_code import StationCode

        old_code = self.code_set.filter(system__code=system_code)
        if old_code:
            old_code = old_code[0]
            old_code.code = code
            old_code.save()
        else:
            system = CodeSystem.objects.get(code=system_code)
            StationCode(system=system, station=self, code=code).save()

    def del_code(self, system):
        self.code_set.get(system__code=system).delete()

    @classmethod
    def get_by_code(cls, system, code):
        if system == 'yandex':
            return cls.objects.get(id=code)

        return cls.objects.get(code_set__system__code__iexact=system,
                               code_set__code__iexact=code)

    @staticmethod
    def get_dict_by_code_system(system):
        from travel.rasp.library.python.common23.models.core.geo.station_code import StationCode

        return {
            sc.code: sc.station
            for sc in StationCode.objects.filter(system__code=system)
                .select_related('station__settlement')
        }

    def save(self, **kwargs):
        self.sirena_id = self.sirena_id or None
        self.time_zone = self.time_zone or None
        self.use_direction = self.use_direction or None
        if not self.country:
            if self.settlement and self.settlement.country:
                self.country = self.settlement.country
            elif self.region and self.region.country:
                self.country = self.region.country

        if not self.time_zone:
            if self.region_id:
                self.time_zone = self.region.time_zone

            if self.settlement_id:
                self.time_zone = self.settlement.time_zone

        if not self.slug:
            self.slug = make_station_slug(self)
        else:
            self.slug = self.slug.lower()

        # on first save
        if not self.id and self.station_type_id == 12 and self.t_type_id:
            self.set_default_station_type()

        super(BaseStation, self).save(**kwargs)
        # если это insert, то иногда уникальный slug можно получить только при известном id
        if not self.slug:
            self.slug = make_station_slug(self)
            if self.slug:
                super(BaseStation, self).save()

    def set_default_station_type(self):
        from travel.rasp.library.python.common23.models.core.geo.station_type import StationType

        try:
            self.station_type = StationType.objects.get(default_for_t_type=self.t_type_id)
        except StationType.DoesNotExist:
            pass

    def __unicode__(self):
        sett = self.settlement and (u" " + self.settlement.__unicode__()) or ""
        reg = u""
        if not sett and self.region:
            reg = u" (" + self.region.title + u")"
        return self.L_title() + u" (%s%s)" % (self.station_type.L_name(), sett or reg)

    @property
    def name(self):
        return self.title

    def get_title_with_prefix(self):
        if self.t_type.code == 'plane':
            return u'а/п ' + self.title

        return self.title

    def get_title(self, rtstation=None, terminal=None):
        terminal = rtstation and rtstation.terminal or terminal

        if terminal:
            return "%s-%s" % (self.title, terminal.name)

        return self.title

    def L_title_with_terminal(self, rtstation=None, terminal=None):
        terminal = rtstation and rtstation.terminal or terminal

        if terminal:
            return "%s-%s" % (self.L_title(), terminal.name)

        return self.L_title()

    def get_popular_title(self, terminal=None, rtstation=None, suburban=False):
        title = self.L_popular_title()
        terminal = rtstation and rtstation.terminal or terminal

        if terminal:
            return "%s-%s" % (title, terminal.name)

        if suburban and self.t_type.code == 'plane':
            return gettext(u"а/п %s") % title

        return title

    def get_short_title(self, suburban=False):
        if suburban and self.t_type.code == 'plane':
            return u"а/п %s" % (self.short_title or self.title)

        return self.short_title or self.title

    def L_short_title_with_prefix(self, suburban=False):
        if suburban and self.t_type.code == 'plane':
            return gettext(u"а/п %s") % (self.L_short_title() or self.L_title())

        return self.L_short_title() or self.L_title()

    def get_clean_title(self):
        return RE_MDOT.sub('', self.get_popular_title())

    def L_clean_title(self):
        return RE_MDOT.sub('', self.L_popular_title_extra())

    def extended_title(self):
        title = u"%s&nbsp;%s" % (self.station_type.L_prefix(), self.title)

        if self.settlement_id and self.t_type_id != 2:
            title += u" (г.&nbsp;%s)" % self.settlement.title

        if self.region_id and not self.settlement_id and self.t_type_id != 2:
            title += u" (обл.&nbsp;%s)" % self.region.title

        return title

    @property
    def is_metro(self):
        return bool(RE_MDOT.match(self.get_popular_title()))

    def L_title_with_prefix(self, lang=None):
        prefix = self.station_type and self.station_type.L_prefix(lang=lang)

        title = self.L_title(lang=lang)

        if prefix:
            return u'%s %s' % (prefix, title)
        else:
            return title

    def L_title_with_type(self, lang=None):
        lang = lang or get_language()

        if lang == 'tr':
            return self.L_title_with_type_tr()

        return self.L_title_with_prefix(lang)

    def L_title_with_type_tr(self):
        title = self.L_title(lang='tr')

        type_title = self.station_type and self.station_type.name_tr

        if type_title:
            return '%s %s' % (title, type_title)

        return title

    def L_title_with_railway_prefix(self, lang=None):
        prefix = self.station_type and self.station_type.L_railway_prefix(lang=lang)

        title = self.L_title(lang=lang)

        if prefix:
            return u'%s %s' % (prefix, title)

        else:
            return title

    @property
    def disputed_territory(self):
        return bool(
            (self.settlement_id and self.settlement.disputed_territory) or
            (self.region_id and self.region.disputed_territory) or
            (self.district_id and self.district.disputed_territory)
        )

    def L_omonim_title(self, lang=None, show_district=True, national_version=None):
        """ Название станции для саджестов """
        result = {
            'title': self.L_title_with_prefix(lang=lang),
            'add': '',
        }
        add = []

        if show_district and self.district:
            add.append(self.district.L_title_with_postfix(lang=lang))

        if self.settlement_id:
            add.append(self.settlement.L_title(lang=lang))

        if self.region_id and (self.t_type_id == 2 or not self.settlement_id):
            add.append(self.region.L_title(lang=lang))

        if self.t_type_id == 2 or not (self.settlement_id or self.region_id):
            country = self.translocal_country(national_version)
            if country:
                add.append(country.L_title(lang=lang))

        if hasattr(self, 'force_show_region') and self.region_id:
            add.append(self.region.L_title(lang=lang))

        direction = self._get_direction_title(lang=lang, national_version=national_version)
        if direction:
            add.append(direction)

        if add:
            result['add'] = u", ".join(add)

        return result

    def _get_direction_title(self, lang=None, national_version=None):
        if hasattr(self, 'show_direction') and national_version != 'tr':
            try:
                direction = self.externaldirectionmarker_set.all().select_related('external_direction')[
                    0].external_direction
            except IndexError:
                return
            else:
                # Не могу определить город направления
                return direction.L_cutted_full_title(lang=lang)

    def L_omonim_title_bem(self, request, direction, **extra_params):
        """Название станции со ссылкой для переключения в омонимах"""

        result = {
            'title': self.L_title_with_prefix(),
        }

        add = []

        if self.district:
            add.append(u"%s р-н" % self.district.title)

        if self.settlement_id:
            if hasattr(self, 'hide_settlement') or extra_params.get('hide_settlement'):
                add.append(self.settlement.L_title())
            else:
                add.append({
                    'block': 'b-link',
                    'url': get_params(request.GET, **{direction + 'Id': self.settlement.point_key,
                                                      direction + 'Name': self.settlement.L_title()}),
                    'content': self.settlement.L_title_with_prefix(),
                })

        if self.region_id and (self.t_type_id == 2 or not self.settlement_id):
            add.append(self.region.L_title())

        if (
            self.country_id and
            (self.t_type_id == 2 or not (self.settlement_id or self.region_id)) and
            not self.disputed_territory
        ):
            add.append(self.country.L_title())

        if hasattr(self, 'force_show_region') and self.region_id:
            add.append(self.region.L_title())

        if hasattr(self, 'show_direction'):
            try:
                direction = self.externaldirectionmarker_set.all().select_related('external_direction')[
                    0].external_direction
            except IndexError:
                pass
            else:
                s_id = self.settlement_id or (direction.suburban_zone and direction.suburban_zone.settlement_id) or None

                if s_id:
                    add.append({
                        'block': 'b-link',
                        'url': composeurl('city_direction', args=[s_id], params={'direction': direction.code}),
                        'title': cut_direction(direction.full_title),
                        'content': cut_direction(direction.full_title)
                    })
                else:
                    # Не могу определить город направления
                    add.append(cut_direction(direction.full_title))

        if add:
            result['add'] = add

        return result

    def full_title(self, link=None, terminal=None):
        from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement

        u"""Домодедово, г. Москва, Московская область, Россия"""
        if link:
            result = u'<a href="%s">%s</a>' % (link, self.get_title(terminal=terminal))
        else:
            result = self.get_title(terminal=terminal)

        if self.settlement:
            result += ', ' + self.settlement.title_with_prefix

        if self.region and not (self.settlement and self.settlement.id == Settlement.MOSCOW_ID):
            result += u", %s" % self.region.title

        if self.country and self.country.id != 225:
            result += u", %s" % self.country.title

        return result

    def L_full_geography(self, show_district=False, national_version=None, lang=None):
        breadcrumbs = [
            self.settlement and self.settlement.L_title(),
            show_district and self.district and self.district.title,
            self.region and self.region.L_title(),
            self._get_direction_title(lang=lang, national_version=national_version)
        ]

        country = self.translocal_country(national_version)

        if country:
            breadcrumbs.append(country and country.L_title())

        return u", ".join(b for b in breadcrumbs if b)

    def L_suggest_full_title_tr(self, lang=None):
        """Full title for turkish suggests"""
        from travel.rasp.library.python.common23.models.core.geo.city_majority import CityMajority

        override = TR_SUGGEST_FULL_TITLE_OVERRIDES.get(self.point_key)

        if override:
            return override

        parts = [
            self.L_title_with_type(lang),
        ]

        if self.settlement_id:
            parts.append(self.settlement.L_title(lang=lang))

            if self.region_id and self.settlement.majority_id > CityMajority.REGION_CAPITAL_ID:
                parts.append(self.region.L_title(lang=lang))

        else:
            if self.region_id:
                parts.append(self.region.L_title(lang=lang))

        country = self.translocal_country('tr')

        if country:
            parts.append(country.L_title(lang=lang))

        return parts

    @property
    def full_address(self):
        return self.make_full_address(self.L_address())

    def make_full_address(self, address):
        if not address:
            return

        elements = [
            self.show_settlement and self.settlement and self.settlement.L_title(),
            address,
            self.near_metro and u'м. ' + self.near_metro or ''
        ]

        return u", ".join(el for el in elements if el)

    def get_nbsp_full_address(self, lang=None):
        lang = lang or get_language()

        address = getattr(self, 'address_{}'.format(lang), None)

        is_common_address = False

        if not address:
            is_common_address = True
            address = self.address or u''

        full_address = self.make_full_address(address)

        if is_common_address or lang == 'ru':
            full_address = self.nbsp_ru_address(full_address)

        return full_address

    @classmethod
    def nbsp_ru_address(cls, address):
        return re.sub(r'(^|\s)(пл|пр|ул|тр|д|к|кв)\.\s+', r'\1\2.&nbsp;', address or u'', re.U)

    def get_country_id(self):
        from travel.rasp.library.python.common23.models.core.geo.express2country import Express2Country

        if self.country_id:
            return self.country_id

        if self.region_id:
            return self.region.country_id

        if self.settlement_id:
            if self.settlement.country_id:
                return self.settlement.country_id

            return self.settlement.region.country_id

        if self.express_id:
            return Express2Country.get_country(self.express_id)

    def is_foreign(self, national_version=None):
        if national_version == 'tr' and self.settlement and self.settlement.id in NOTHERN_CYPRUS_CITIES:
            return False
        return self.get_country_id() not in settings.OUR_COUNTRIES

    def in_local_kubr(self, lang=None):
        # RASP-11686
        if not lang:
            lang = translation.get_language()

        if lang not in settings.LOCAL_KUBR:
            lang = 'ru'

        country_ids = settings.LOCAL_KUBR.get(lang, [])

        return self.get_country_id() in country_ids

    @property
    def has_info(self):
        """ Станция с информацией"""
        return bool(self.L_address() or
                    self.how_to_get_to_city_ru or
                    self.site_url or
                    self.properties.count() or
                    self.stationphone_set.count() or
                    (self.latitude and self.longitude))

    def interesting(self):
        """ Станция представляет хоть какой-то интерес """
        return (self.country_id or self.settlement_id or self.region_id) and \
               (self.majority_id < 5 or self.has_info)

    def is_empty(self):
        """ Станция без чего-либо интересного """
        return not (self.majority_id < 5 or self.has_info)

    def has_actual_tablo(self):
        """Табло станции обновляется регулярно"""
        return self.tablo_state == 'real'

    @property
    def in_tablo(self):
        """Табло строится для станции"""
        from travel.rasp.library.python.common23.models.core.geo.station_majority import StationMajority

        return self.settlement_id and (self.majority_id <= StationMajority.IN_TABLO_ID)

    def not_in_search(self):
        """ Важность станции - "не в поиске" или ниже """
        from travel.rasp.library.python.common23.models.core.geo.station_majority import StationMajority

        return not self.majority_id or self.majority_id >= StationMajority.objects.get(code='not_in_search').id

    def not_in_search_exact_mta(self):
        """ Важность станции - "не в поиске" и станция относится к МТА"""
        from travel.rasp.library.python.common23.models.core.geo.station_majority import StationMajority

        return self.majority_id == StationMajority.objects.get(code='not_in_search').id and self.is_mta

    @property
    def show_link(self):
        return (
            (not self.not_in_search() or (self.majority_id == 5 and self.is_mta)) and
            (self.country_id or self.settlement_id or self.region_id) and
            (self.has_info or (self.majority_id < 5 or (self.majority_id == 5 and self.is_mta))) and
            not self.hidden
        )

    @property
    def is_mta(self):
        # TODO: метод не должен делать джоины по другим таблицам. Видимо, нужно прописывать этот
        # признак станции при пересчете
        return False

        # МТА станции определяем по соответствию импорта или по рейсам
        return (self.stationmapping_set.filter(supplier__code='mta').count() > 0 or
                self.rtstation_set.filter(thread__route__supplier__code='mta').count() > 0)

    @property
    def title_with_prefix(self):
        prefix = None

        if self.station_type:
            prefix = self.station_type.L_prefix()

        if prefix:
            return u'%s %s' % (prefix, self.title)

        return self.title

    @property
    def railway_title(self):
        if self.station_type and self.station_type.L_railway_prefix():
            return self.station_type.L_railway_prefix() + u' ' + self.title

        return self.title

    def L_railway_title(self, rtstation=None, terminal=None, thread=None):
        # RASP-8900 на странице автобусного рейса МТА отображать город для станций
        if thread and thread.supplier.code == 'mta' and self.settlement and self.settlement.majority_id <= 4:
            # TODO: заменить _norm на normalize из models.precalc и использовать локальные названия при сравнении
            def equivalent(q, p):
                def _norm(v):
                    return v.lower().replace(u'ё', u'е')

                return _norm(q) == _norm(p)

            if self.not_generalize and not equivalent(self.settlement.title, self.title):
                return u'%s (%s)' % (self.L_title(), self.settlement.L_title())

            else:
                return u'%s' % self.settlement.L_title()

        title = self.L_title_with_railway_prefix()

        terminal = rtstation and rtstation.terminal or terminal

        if terminal:
            return "%s-%s" % (title, terminal.name)

        return title

    @property
    def suburban_title(self):
        if self.t_type.code == 'plane':
            return u"а/п %s" % self.title

        return self.title

    def L_suburban_title(self, lang=None):
        title = self.L_title(lang=lang)

        if self.t_type.code == 'plane':
            return gettext(u"а/п %s", lang=lang) % title

        return title

    @property
    def title_ru_phrase_from(self):

        if self.station_type.name_ru == u'аэропорт':
            return u'из аэропорта %s' % self.L_title(lang='ru')
        else:
            return super(BaseStation, self).title_ru_phrase_from

    @property
    def title_ru_phrase_to(self):

        if self.station_type.name_ru == u'аэропорт':
            return u'в аэропорт %s' % self.L_title(lang='ru')
        else:
            return super(BaseStation, self).title_ru_phrase_to

    def has_suburban_traffic(self):
        type_choices = self.type_choices_set

        if 'suburban' in type_choices or 'aeroex' in type_choices:
            return True

        return False

    @property
    def iata(self):
        from travel.rasp.library.python.common23.models.core.geo.station import Station

        try:
            return self._iata
        except AttributeError:
            pass

        if Station.code_manager.code_cache:
            self._iata = Station.code_manager.getcodes(self).get('iata')
        else:
            self._iata = self.get_code('iata')

        return self._iata

    @property
    def icao(self):
        return self.get_code('icao')

    @property
    def buscomua(self):
        try:
            return self._buscomua
        except AttributeError:
            self._buscomua = self.get_code('buscomua')
            return self._buscomua

    @property
    def export_uid(self):
        u"""ID станции для экспорта"""
        return "%s__%d" % (self.export_type, self.id)

    @property
    def export_type(self):
        u"""Тип станции для экспорта"""
        if self.t_type_id != 8:
            return 'station'

        return 'station' if self.is_subway else 'stop'

    # Случайная выборка названий станций для форм
    _sample_titles = None

    @classmethod
    def precache_sample_titles(cls):
        lang = translation.get_language()

        # RASP-9691
        title_field = 'title'

        if lang in ('tr', 'uk'):
            title_field += '_%s' % lang

        stations = cls.objects.filter(t_type=1, hidden=False).values_list(title_field, 'region_id', 'country_id')

        cls._sample_titles = {
            'by_region_id': {},
            'by_country_id': {},
        }

        for title, r_id, c_id in stations:
            if r_id:
                cls._sample_titles['by_region_id'].setdefault(r_id, []).append(title)

            if c_id:
                cls._sample_titles['by_country_id'].setdefault(c_id, []).append(title)

    @classmethod
    def sample_title(cls, region_id, country_id):
        titles = None

        if not cls._sample_titles:
            cls.precache_sample_titles()

        if region_id:
            titles = cls._sample_titles['by_region_id'].get(region_id)

        if not titles and country_id:
            titles = cls._sample_titles['by_country_id'].get(country_id)

        if titles:
            return random.choice(titles)

    def get_direction(self):
        from travel.rasp.library.python.common23.models.core.directions.station2direction import Station2Direction
        return Station2Direction.get_direction(self)

    @property
    def tz_name_abbr(self):
        if self.settlement:
            tz_name = self.settlement.title_abbr or self.settlement.title_in or self.title_ru_locative or self.title
        else:
            tz_name = self.title_ru_locative or self.title

        return u'время в %s' % tz_name

    def L_info_title(self):
        popular_title = self.L_popular_title()

        if popular_title:
            return popular_title

        prefix = self.station_type.L_prefix()

        if prefix:
            return u"%s %s" % (prefix, nbsp(self.L_title()))

        return nbsp(self.L_title())

    def get_schedule_url(self, type_, direction=None, subdir=None):
        params = {
            'span': 'schedule',
            'type': type_
        }

        if direction:
            params['direction'] = direction

        if subdir:
            params['subdir'] = subdir

        return composeurl('station', args=[self.id], params=params)

    def get_zoom(self):
        if self.map_zoom:
            return self.map_zoom

        if self.settlement and self.settlement.has_ymap and self.t_type.code != 'plane':
            return 13

        return 10

    def determine_station_type(self):
        from travel.rasp.library.python.common23.models.core.geo.station_type import StationType
        from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement

        types = StationType.types_map()

        if self.t_type_id == 1:
            # Поезда
            for t, regexp in StationType.regex_list():
                if regexp.match(self.title):
                    return t

            try:
                if self.settlement and self.majority_id <= 2:
                    return types[u'вокзал']
            except Settlement.DoesNotExist:
                pass

            if not self.station_type or self.station_type_id == 12:
                return types[u'станция']
            else:
                return self.station_type

        elif self.t_type_id == 2:
            # Самолеты
            return types[u'аэропорт']

        elif self.t_type_id == 3:
            # Автобусы
            if self.settlement and self.settlement.majority_id <= 4:
                return types[u'автовокзал']

            else:
                return types[u'автобусная остановка']

        return types[u'станция без типа']

    def modify_title(self):
        u"""Изменения названия согласно типу станции"""
        from travel.rasp.library.python.common23.models.core.geo.station_type import StationType

        if not self.t_type_id == 1:
            return

        regexes = StationType.regex_list()
        for t, regexp in regexes:
            m = regexp.match(self.title)
            if m:
                self.station_type = t

                # Берем остаток после префикса
                _, end = m.span()
                self.title = self.title[end:].capitalize()

                # Это заполнит склонятор потом
                self.title_ru_preposition_v_vo_na = self.title_ru_locative = None
                self.title_ru_accusative = self.title_ru_genitive = None
                self.save()

    def as_xml(self, code_system=None):
        element = etree.Element('stoppoint',
                                name=self.title,
                                lat=str(self.latitude),
                                lon=str(self.longitude),
                                timezone=self.time_zone or '')

        if self.settlement:
            element.set('settlement_id', str(self.settlement.id))

        if self.country or self.settlement.country:
            element.set('country_id', str((self.country or self.settlement.country).id))

        if self.region or self.settlement.region:
            element.set('region_id', str((self.region or self.settlement.region).id))

        if self.t_type:
            element.set('t_type', self.t_type.code)

        if self.supplier:
            element.set('supplier', self.supplier.code)

        if code_system is not None:
            if self.get_code(code_system):
                element.set('vendor', code_system)
                element.set('vendor_id', self.get_code(code_system))
            else:
                element.set('station_id', str(self.id))

        exits = etree.Element('terminals')
        for t in self.stationterminal_set.all():
            exits.append(t.as_xml())

        if len(exits) > 0:
            element.append(exits)

        return element

    def as_point(self):
        return self.latitude, self.longitude

    def L(self, field, case='nominative', lang=None):
        return self.linguistics(field, lang or translation.get_language(), case)

    def L_popular_title_extra(self, terminal=None, rtstation=None, suburban=False):
        # FIXME: выпилить эту функцию, заменив ее вызовы на get_popular_title
        title = self.L_popular_title()

        terminal = rtstation and rtstation.terminal or terminal

        if terminal:
            return "%s-%s" % (title, terminal.name)

        if suburban and self.t_type.code == 'plane':
            return xgettext(u"а/п <airport-title/>", airport_title=title)

        return title

    def get_local_language(self):
        return self.country and self.country.language

    # TODO: убрать, использовать i18n.tanker{download|upload}
    @classmethod
    def tankerupload(cls, tanker, stations, keyset):
        keys = {}

        for station in stations.select_related('country'):
            translations = {}

            for lang in L_field.LANGUAGES:
                form = getattr(station, 'title_%s' % lang)

                if form:
                    translations[lang] = {
                        "form": form,
                    }

            translations[station.country.language] = {
                "form": station.title
            }

            keys[str(station.id)] = {
                "info": {
                    "is_plural": False,
                    "content": "%s (%s)" % (station.title, station.country.language)
                },

                "translations": translations
            }

        tanker.upload(keyset, keys, L_field.LANGUAGES)

    @classmethod
    def tankerdownload(cls, tanker):
        updated = 0

        for keyset in tanker.list():
            if keyset.startswith('stations_'):
                print(smart_str(
                    u'Downloading %s[%s]' % (
                        force_text(keyset),
                        ','.join(force_text(lang) for lang in L_field.LANGUAGES),
                    )
                ))

                keys = tanker.download(keyset, L_field.LANGUAGES)

                stations = cls.objects.filter(pk__in=list(keys.keys()))

                for station in stations:
                    fields_updated = 0

                    key = keys[str(station.id)]

                    for lang in L_field.LANGUAGES:
                        override = getattr(station, 'title_%s_override' % lang)

                        if not override:
                            form = key['translations'][lang]['form']

                            old_form = getattr(station, 'title_%s' % lang)

                            if form and old_form != form:
                                setattr(station, 'title_%s' % lang, form)

                                updated += 1
                                fields_updated += 1

                    if fields_updated:
                        station.save()

        print(smart_str('Updated %d stations' % updated))


class StationProperty(models.Model):
    title = models.CharField(verbose_name=_(u'наименование'), max_length=100)
    description = models.TextField(verbose_name=_(u'описание'),
                                   null=True, blank=True, default=None)
    order = models.IntegerField(default=10, help_text=_(u"чем больше цифра, тем хуже признак"))

    def __unicode__(self):
        return self.title

    class Meta(object):
        verbose_name = _(u'свойство станции')
        verbose_name_plural = _(u'свойства станций')
        ordering = ('title',)
        app_label = 'www'
        db_table = 'www_stationproperty'
