# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.conf.global_settings import LANGUAGES
from django.contrib.contenttypes.fields import GenericRelation
from django.db import models
from django.utils.lru_cache import lru_cache
from django.utils.translation import ugettext_lazy as _

from travel.rasp.library.python.common23.models.core.geo.city_majority import CityMajority
from travel.rasp.library.python.common23.models.core.geo.point import Point
from travel.rasp.library.python.common23.models.fields import CodeCharField
from travel.rasp.library.python.common23.models.precache.manager import PrecachingManager
from travel.rasp.library.python.common23.models.texts.geobase_ltitle import Geobase_L_title
from travel.rasp.library.python.common23.models.texts.i18n import L_field


@lru_cache()
def get_country_capital(country):
    from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement

    return Settlement.objects.filter(country=country, majority__id=CityMajority.CAPITAL_ID).first()


class Country(models.Model, Point):
    """ Страна """

    L_title = Geobase_L_title(_(u'наименование'), add_local_field=True, local_field_critical=True, extra={
        'ru': [
            ('preposition_v_vo_na', L_field.char_field(_(u'предлог (в, во, на)'), max_length=2)),  # prefix_in
            ('genitive', L_field.char_field(_(u'наименование (род. падеж'))),  # title_of
            ('locative', L_field.char_field(_(u'наименование (местный. падеж)')))  # title_in
        ],
        'uk': [
            # В геобазе нет винительного падежа у стран
            ('accusative', L_field.char_field(_(u'наименование (вин. падеж) (uk)'))),
        ]
    })

    _geo_id = models.IntegerField(verbose_name=_(u'geo ID'), null=True,
                                  blank=True, default=None, unique=True)
    _kladr_id = models.CharField(verbose_name=_(u'kladr ID'), max_length=50,
                                 null=True, blank=True, default=None,
                                 editable=False)
    code = CodeCharField(verbose_name=_(u'код страны ISO 3166-1 2-alpha'), max_length=4,
                         null=True, blank=True, default=None, unique=True)
    code3 = CodeCharField(verbose_name=_(u'код страны ISO 3166-1 3-alpha'), max_length=4,
                          null=True, blank=True, default=None, unique=True)
    domain_zone = CodeCharField(verbose_name=_(u'имя доменной зоны IANA'), max_length=2,
                                null=True, blank=True, default=None, unique=True,
                                help_text=_(u"без точки: 'ru', 'us', 'uk'"))
    synonyms = GenericRelation('PointSynonym')

    currency = models.ForeignKey('currency.Currency', verbose_name=_(u'валюта'),
                                 null=True, blank=True, default=None)

    language = CodeCharField(
        _(u'язык страны'),
        max_length=5,
        choices=[(k, _(v)) for k, v in LANGUAGES],
        default='ru',
    )
    modified_at = models.DateTimeField(_(u'Дата-время изменения'), auto_now=True, null=False, blank=True)

    objects = PrecachingManager(keys=['pk', 'domain_zone', '_geo_id'])

    RUSSIA_ID = 225
    BELARUS_ID = 149
    UKRAINE_ID = 187
    TAJIKISTAN_ID = 209
    TURKEY_ID = 983
    KAZAKHSTAN_ID = 159
    UZBEKISTAN_ID = 171
    LITVA_ID = 117
    FINLAND_ID = 123

    def __unicode__(self):
        return self.L_title()

    def save(self, **kwargs):
        super(Country, self).save(**kwargs)

    def get_local_language(self):
        return self.language

    @property
    def name(self):
        return self.__unicode__()

    @property
    def title_with_prefix(self):
        return self.title

    @property
    def hidden(self):
        return False

    @property
    def suburban_zone_id(self):
        return None

    def get_capital(self):
        return get_country_capital(self)

    def get_capital_tz(self):
        capital = self.get_capital()
        return capital and capital.time_zone

    @classmethod
    def mast_have_region_countries(cls):
        return [cls.objects.get(code=u"RU"), cls.objects.get(code=u"KZ"),
                cls.objects.get(code=u"US")]

    def L_omonim_title(self, show_district=True, lang=None, national_version=None):
        return {'title': self.L_title(lang=lang), 'add': ''}

    def L_omonim_title_bem(self, request, direction, **extra_params):
        return {'title': self.L_title()}

    def L_suggest_full_title_tr(self, lang=None):
        """Full title for turkish suggests"""

        return [self.L_title(lang=lang)]

    def get_popular_title(self, lang=None):
        # необходимо для общего интерфейса
        return self.L_title(lang=lang)

    def get_railway_tz_city(self):
        """
        :return: ЖД Столицу или None, если у страны нет столицы(встречается в тестах).
        :rtype: Settlement | None
        """
        # В Таджикистане на ЖД московское время
        if self.id == self.TAJIKISTAN_ID:
            country = Country.objects.get(id=self.RUSSIA_ID)
        else:
            country = self

        return country.get_capital()

    def get_sign(self):
        return self.L_title(lang='en').replace(' ', '')

    @classmethod
    def find_by_sign(cls, sign):
        countries = cls.objects.all()
        by_slug = dict([(c.get_sign().lower(), c) for c in countries])
        return by_slug.get(sign.replace(' ', '').lower())

    class Meta(object):
        verbose_name = _(u'страна')
        verbose_name_plural = _(u'страны')
        app_label = 'www'
        db_table = 'www_country'
