# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import warnings

from django.conf import settings
from django.core.exceptions import ObjectDoesNotExist
from django.utils import translation

from travel.rasp.library.python.common23.utils.caching import cached
from travel.rasp.library.python.common23.utils.text import title_with_preposition
from travel.rasp.library.python.common23.utils.warnings import RaspDeprecationWarning


class Point(object):
    @property
    def is_station(self):
        from travel.rasp.library.python.common23.models.core.geo.base_station import BaseStation
        return isinstance(self, BaseStation)

    @classmethod
    def model_key_prefix(cls, model):
        from travel.rasp.library.python.common23.models.core.geo.base_station import BaseStation
        from travel.rasp.library.python.common23.models.core.geo.country import Country
        from travel.rasp.library.python.common23.models.core.geo.region import Region
        from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement

        if issubclass(model, Settlement):
            return 'c'

        elif issubclass(model, BaseStation):
            return 's'

        elif issubclass(model, Region):
            return 'r'

        elif issubclass(model, Country):
            return 'l'

        raise NotImplementedError('Keys for type %s are not supported' % model.__name__)

    @classmethod
    def get_point_key(cls, model, pk):
        prefix = cls.model_key_prefix(model)

        return '%s%s' % (prefix, pk)

    @property
    def point_key(self):
        return self.get_point_key(self.__class__, self.pk)

    @classmethod
    def parse_key(cls, key):
        from travel.rasp.library.python.common23.models.core.geo.country import Country
        from travel.rasp.library.python.common23.models.core.geo.region import Region
        from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement
        from travel.rasp.library.python.common23.models.core.geo.station import Station

        try:
            prefix, pk = key[0], key[1:]

        except (IndexError, TypeError):
            raise ValueError('Unknown key %r' % key)

        if prefix == 'c':
            return Settlement, pk
        elif prefix == 's':
            return Station, pk
        elif prefix == 'r':
            return Region, pk
        elif prefix == 'l':
            return Country, pk

        raise ValueError('Unknown type prefix %s' % prefix)

    @classmethod
    def get_by_key(cls, key, use_hidden_manager=True):
        from travel.rasp.library.python.common23.models.core.geo.country import Country
        from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement

        try:
            prefix, geo_id = key[0], key[1:]
        except (IndexError, TypeError):
            raise ValueError('Unknown key %r' % key)

        if prefix == 'g':
            try:
                return Settlement.objects.get(_geo_id=geo_id)
            except Settlement.DoesNotExist:
                pass

            try:
                return Country.objects.get(_geo_id=geo_id)
            except Country.DoesNotExist:
                pass

            raise ObjectDoesNotExist('Geo Object with geo_id == %s does not exist' % geo_id)

        model, pk = cls.parse_key(key)

        if use_hidden_manager and hasattr(model, 'hidden_manager'):
            return model.hidden_manager.get(pk=pk)

        return model.objects.get(pk=pk)

    @classmethod
    def in_bulk(cls, keys):
        foo = {}

        for key in keys:
            model, pk = cls.parse_key(key)

            foo.setdefault(model, []).append(pk)

        objects = {}

        for model, pks in foo.items():
            prefix = cls.model_key_prefix(model)

            for obj in model.objects.in_bulk(pks).values():
                objects['%s%s' % (prefix, obj.pk)] = obj

        return objects

    def L_title_phrase_from(self, lang=None):
        """
        Получает локализованную фразу 'из города': 'Из Киева', 'from Kiev', 'з Києва'.
        Если не найдется название города в подходящем падеже, то вернется None.
        """
        lang = lang or translation.get_language()

        if lang == 'tr':
            return None

        if lang == 'en':
            title = self.L_title(lang=lang)
            return title_with_preposition('from', title)

        if lang == 'ru':
            return self.title_ru_phrase_from

        if lang == 'uk':
            title = self.L_title(case='genitive', fallback=False, lang=lang)
            return title_with_preposition('з', title)

    def L_title_phrase_to(self, lang=None):
        """
        Получает локализованную фразу 'в город': 'В Киев', 'to Kiev', 'до Києва'.
        Если не найдется название города в подходящем падеже, то вернется None.
        """
        lang = lang or translation.get_language()

        if lang == 'tr':
            return None

        if lang == 'en':
            title = self.L_title(lang=lang)
            return title_with_preposition('to', title)

        if lang == 'ru':
            return self.title_ru_phrase_to

        if lang == 'uk':
            title = self.L_title(case='genitive', fallback=False, lang=lang)
            return title_with_preposition('до', title)

        return None

    @property
    def title_ru_phrase_from(self):
        title = self.L_title(lang='ru', fallback=False, case='genitive')
        return title_with_preposition('из', title)

    @property
    def title_ru_phrase_in(self):
        title = self.L_title(lang='ru', case='locative')
        return title_with_preposition(self.title_ru_preposition_v_vo_na or 'в', title)

    @property
    def title_ru_phrase_to(self):
        title = self.L_title(lang='ru', fallback=False, case='accusative')
        return title_with_preposition(self.title_ru_preposition_v_vo_na or 'в', title)

    @property
    @cached(lambda self: 'find_iata_point_%s%s' % (self.__class__.__name__, self.id),
            timeout=settings.CACHES['default']['LONG_TIMEOUT'])
    def iata_point(self):
        from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement
        from travel.rasp.library.python.common23.models.core.geo.station import Station

        if isinstance(self, Station):
            station = self
            settlement = self.settlement
            if settlement and settlement.iata:
                return settlement
            if station.iata:
                return station

        elif isinstance(self, Settlement):
            stations = self.station_set.filter(t_type__code='plane', hidden=False)
            if stations:
                # У города есть свои аэропорты, берем его код если указан
                settlement = self
                if settlement and settlement.iata:
                    return settlement
                for station in stations:
                    if station.iata:
                        return station
            else:
                # Если до города летают, то должен быть хотя бы один аэропорт,
                # привязанный на прямую или через множественную привязку.
                # Потому игнорируем наличие у города код iata
                # Если он не привязан ни к одному аэропорту
                s2s_set = self.station2settlement_set \
                    .filter(station__t_type__code='plane', station__hidden=False) \
                    .select_related('station')
                for s2s in s2s_set:
                    station = s2s.station
                    settlement = station.settlement
                    if settlement and settlement.iata:
                        return settlement
                    if station.iata:
                        return station

        return None

    @property
    @cached(lambda self: 'find_sirena_point_%s%s' % (self.__class__.__name__, self.id),
            timeout=settings.CACHES['default']['LONG_TIMEOUT'])
    def sirena_point(self):
        from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement
        from travel.rasp.library.python.common23.models.core.geo.station import Station

        """ Ищет точку с не пустым sirena_id, например, аэропорт города, если у
        города sirena_id не указан. Точка может быть привязана, как напрямую, так и
        через множественную связь.
        Если это аэропорт, то постараемся взять город для этого аэропорта
        """

        if isinstance(self, Station):
            station = self
            settlement = self.settlement
            if settlement and settlement.sirena_id:
                return settlement
            if station.sirena_id:
                return station

        elif isinstance(self, Settlement):
            stations = self.station_set.filter(t_type__code='plane', hidden=False)
            if stations:
                # У города есть свои аэропорты, берем его код если указан
                settlement = self
                if settlement and settlement.sirena_id:
                    return settlement
                for station in stations:
                    if station.sirena_id:
                        return station
            else:
                # Если до города летают, то должен быть хотя бы один аэропорт,
                # привязанный на прямую или через множественную привязку.
                # Потому игнорируем наличие у города sirena_id
                # Если он не привязан ни к одному аэропорту
                s2s_set = self.station2settlement_set \
                    .filter(station__t_type__code='plane', station__hidden=False) \
                    .select_related('station')
                for s2s in s2s_set:
                    station = s2s.station
                    settlement = station.settlement
                    if settlement and settlement.sirena_id:
                        return settlement
                    if station.sirena_id:
                        return station

        return None

    def get_ukrmintrans_express_codes(self):
        stations = self.get_express_stations()

        return [st.express_id for st in stations]

    @cached(lambda self: 'get_express_stations%s%s' % (self.__class__.__name__, self.id),
            timeout=settings.CACHES['default']['LONG_TIMEOUT'])
    def get_express_stations(self):
        """
        Для станции возвращает список из нее, если у нее есть экспресс код, иначе пустой список.
        Для города возвращает список из фейковых станций express,
        если нет фейковых express станций, то возвращает список из главных станций в городе с экспресс кодами,
        если нет главных станций, то возвращает список всех станций с экспресс кодами.
        """
        from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement
        from travel.rasp.library.python.common23.models.core.geo.station import Station

        if isinstance(self, Station):
            if self.express_id:
                return [self]
            else:
                return []

        elif isinstance(self, Settlement):
            settlement = self

        else:
            return []

        stations = settlement.station_set.filter(hidden=False, express_id__isnull=False) \
            .only('express_id', 'majority__code', 'country')

        express_fake_stations = [st for st in stations if st.majority.code == 'express_fake']

        if express_fake_stations:
            return express_fake_stations

        main_in_city_express_stations = [st for st in stations if st.majority.code == 'main_in_city']

        if main_in_city_express_stations:
            return main_in_city_express_stations

        return stations

    def get_settlement_geo_id(self):
        from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement
        from travel.rasp.library.python.common23.models.core.geo.station import Station

        if isinstance(self, Station):
            return self.settlement_id and self.settlement._geo_id or None
        elif isinstance(self, Settlement):
            return self._geo_id
        return None

    def get_region_geo_id(self):
        from travel.rasp.library.python.common23.models.core.geo.region import Region
        from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement
        from travel.rasp.library.python.common23.models.core.geo.station import Station

        if isinstance(self, Station) or isinstance(self, Settlement):
            return self.region_id and self.region._geo_id or None
        elif isinstance(self, Region):
            return self._geo_id
        return None

    @property
    def url_target(self):
        from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement
        from travel.rasp.library.python.common23.models.core.geo.station import Station

        if isinstance(self, Settlement):
            return 'city'
        elif isinstance(self, Station):
            return 'station'
        else:
            raise NotImplementedError('Urls for type %s are not supported' % self.__class__.__name__)

    def L_short_title(self, *args, **kwargs):
        return self.L_title(*args, **kwargs)

    def L_popular_title(self, *args, **kwargs):
        return self.L_title(*args, **kwargs)

    @property
    def type(self):
        from travel.rasp.library.python.common23.models.core.geo.base_station import BaseStation
        from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement
        from travel.rasp.library.python.common23.models.core.geo.region import Region
        from travel.rasp.library.python.common23.models.core.geo.country import Country

        warnings.warn("[2016-04-21] Don't use point.type.", RaspDeprecationWarning, stacklevel=2)

        if isinstance(self, Settlement):
            return 'settlement'
        elif isinstance(self, BaseStation):
            return 'station'
        elif isinstance(self, Region):
            return 'region'
        elif isinstance(self, Country):
            return 'country'

        raise NotImplementedError('Property "type" for {} is not supported'.format(self.__class__.__name__))
