# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import re

from django.core import validators
from django.db import models
from unidecode import unidecode

from travel.rasp.library.python.common23.models.texts.icao_transliteration import transliterate

POINT_SLUG_LENGTH = 100
SPACE_CHARS_RE = re.compile(r"(\s|\.|\,|;|:|\\|/|\(|\)|'|-|–|\"|»|«|#|<|>|\+|\*|\[|\])+")
VALID_SLUG_RE_STR = r'^([a-z0-9]+_?-?_?)+$'
VALID_SLUG_RE = re.compile(VALID_SLUG_RE_STR)
REPLACE_SYMBOLS = {
    '`': '',
    'ʻ': '',
    '’': '',
    '№': 'N',
    '&': 'and',
    '@': 'a',
    '?': '',
}


def slugify(text):
    text = transliterate(text)
    text = unidecode(text)
    text = SPACE_CHARS_RE.sub('-', text).strip('-')
    text = ''.join([REPLACE_SYMBOLS.get(ch, ch) for ch in text])
    text = text.lower()
    if len(text) > POINT_SLUG_LENGTH:
        text = text[:POINT_SLUG_LENGTH]
    return text


def make_station_slug_variant(station, use_country=False, use_region=False, use_id=False, use_type=False):
    origin = station.L_title(lang='en') or station.L_title(lang='ru')
    if use_type:
        origin += ' {0}'.format(station.station_type.L_name(lang='en') or station.station_type.L_name(lang='ru'))
    if use_region:
        if station.settlement:
            origin += ' {0}'.format(station.settlement.L_title(lang='en') or station.settlement.L_title(lang='ru'))
        elif station.region:
            origin += ' {0}'.format(station.region.L_title(lang='en') or station.region.L_title(lang='ru'))
    if use_country and station.country:
        origin += ' {0}'.format(station.country.code)
    if use_id and station.id:
        origin += ' {0}'.format(station.point_key)
    return slugify(origin)


def make_settlement_slug_variant(settlement, use_country=False, use_region=False, use_id=False, **_kwargs):
    origin = settlement.L_title(lang='en') or settlement.L_title(lang='ru')
    if use_region and settlement.region:
        origin += ' {0}'.format(settlement.region.L_title(lang='en') or settlement.region.L_title(lang='ru'))
    if use_country and settlement.country:
        origin += ' {0}'.format(settlement.country.code)
    if use_id and settlement.id:
        origin += ' {0}'.format(settlement.point_key)
    return slugify(origin)


def make_point_slug_variant(obj, **kwargs):
    from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement
    from travel.rasp.library.python.common23.models.core.geo.station import Station

    if isinstance(obj, Settlement):
        return make_settlement_slug_variant(obj, **kwargs)
    if isinstance(obj, Station):
        return make_station_slug_variant(obj, **kwargs)


# Варианты конфигурации поля slug в порядке увеличения длины и детализации
SETTLEMENT_SLUG_VARIANTS = [
    {'use_country': False, 'use_region': False, 'use_id': False},
    {'use_country': True, 'use_region': False, 'use_id': False},
    {'use_country': False, 'use_region': True, 'use_id': False},
    {'use_country': False, 'use_region': True, 'use_id': True},
]
STATION_SLUG_VARIANTS = [
    {'use_country': False, 'use_region': False, 'use_id': False, 'use_type': False},
    {'use_country': False, 'use_region': False, 'use_id': False, 'use_type': True},
    {'use_country': False, 'use_region': True, 'use_id': False, 'use_type': False},
    {'use_country': False, 'use_region': True, 'use_id': True, 'use_type': False},
]


def make_station_slug(station):
    return make_slug(station, STATION_SLUG_VARIANTS)


def make_settlement_slug(settlement):
    return make_slug(settlement, SETTLEMENT_SLUG_VARIANTS)


def make_slug(obj, slug_variants):
    """Генерирует уникальный текстовый идентификатор обьекта,
    используя первый удачный из переданных вариантов

    :param obj: город или станция
    :param slug_variants: вариатны параметров для функции make_point_slug_variant
    :return: уникальный текстовый идентификатор или None
    """
    slug = make_point_slug_variant(obj, **slug_variants[0])
    # находим все похожие города и станции
    similar_objects = find_all_by_slug_startswith(slug)
    similar_objects = [s for s in similar_objects if obj != s]
    if not similar_objects:
        return slug

    slugs = set(o.slug for o in similar_objects)
    # берем минимальный уровень детализации с уникальным slug
    for args in slug_variants:
        slug = make_point_slug_variant(obj, **args)
        if slug in slugs:
            continue
        for similar_object in similar_objects:
            similar_object_slug = make_point_slug_variant(similar_object, **args)
            slugs.add(similar_object_slug)
            if slug == similar_object_slug:
                break
        else:
            return slug
    return None


def find_by_slug(value):
    from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement
    from travel.rasp.library.python.common23.models.core.geo.station import Station

    try:
        return Settlement.objects.get(slug=value)
    except Settlement.DoesNotExist:
        pass

    try:
        return Station.objects.get(slug=value)
    except Station.DoesNotExist:
        pass


def find_all_by_slug_startswith(value):
    from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement
    from travel.rasp.library.python.common23.models.core.geo.station import Station

    return list(Settlement.objects.filter(slug__startswith=value)) + list(
        Station.objects.filter(slug__startswith=value))


class PointSlugField(models.CharField):
    def pre_save(self, instance, add):
        value = models.CharField.pre_save(self, instance, add)

        if value is not None:
            if not VALID_SLUG_RE.match(value):
                raise validators.ValidationError(
                    'Допустимы только символы латинского алфавита, цифры, дефис и земля. slug={}'.format(value)
                )

            duplicate_instance = find_by_slug(value)
            if duplicate_instance and duplicate_instance != instance:
                raise validators.ValidationError('Значение "{0}" уже используется {1}'.format(
                    value, duplicate_instance.point_key))
            return value
