# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import re

from django.db import models
from django.utils.translation import ugettext_lazy as _

from travel.rasp.library.python.common23.models.core.geo.country import Country
from travel.rasp.library.python.common23.models.core.geo.point import Point
from travel.rasp.library.python.common23.models.core.geo.timezone_mixin import TimeZoneMixin
from travel.rasp.library.python.common23.models.fields import CodeCharField
from travel.rasp.library.python.common23.models.precache.manager import PrecachingManager
from travel.rasp.library.python.common23.models.texts.geobase_ltitle import Geobase_L_title
from travel.rasp.library.python.common23.models.utils import HiddenManagerWrapper


class Region(models.Model, TimeZoneMixin, Point):
    """ Область """

    country = models.ForeignKey(Country, verbose_name=_(u'страна'),
                                null=False, blank=False)

    L_title = Geobase_L_title(_(u'наименование'), add_local_field=True, local_field_critical=True)

    time_zone = models.CharField(verbose_name=_(u'временная зона'),
                                 help_text=_(u'временная зона в международном формате'),
                                 null=True, blank=False, max_length=30)
    _geo_id = models.IntegerField(verbose_name=_(u'geo ID'), null=True,
                                  blank=True, default=None, unique=True)
    _kladr_id = models.CharField(verbose_name=_(u'kladr ID'), max_length=50,
                                 null=True, blank=True, default=None,
                                 editable=False)
    hidden = models.BooleanField(verbose_name=_(u'не показывать нигде!'),
                                 default=False)
    koatuu = CodeCharField(verbose_name=_(u'код КОАТУУ'), max_length=40,
                           null=True, blank=True, default=None, unique=True,
                           help_text=_(u"держа́вний класифіка́тор об'є́ктів"
                                       u" адміністрати́вно-територіа́льного у́строю Украї́ни"))
    agent_geo_id = models.IntegerField(verbose_name=_(u'agent Geo ID'), null=True,
                                       blank=True, default=None)
    disputed_territory = models.BooleanField(verbose_name=_(u"спорная территория"), default=False)
    modified_at = models.DateTimeField(_(u'Дата-время изменения'), auto_now=True, null=False, blank=True)

    not_show_not_agreed_in_yabus = models.BooleanField(
        verbose_name=_(u'не показывать отправления от несогласованных остановок в Яндекс.Автобусах'),
        default=False
    )
    not_show_not_agreed = models.BooleanField(
        verbose_name=_(u'не показывать отправления от несогласованных остановок'),
        default=False
    )
    show_is_from_bus_station = models.BooleanField(
        verbose_name=_(u'показывать признак "рейс от автовокзала"'),
        default=False
    )

    objects = PrecachingManager(keys=['pk', '_geo_id'])
    hidden_manager = HiddenManagerWrapper('objects')

    MOSCOW_REGION_ID = 1
    KALININGRAD_REGION_ID = 10857
    CRIMEA_REGION_ID = 977

    def get_capital(self):
        try:
            return self.settlement_set.filter(majority__id__in=[1, 2])[0]
        except IndexError:
            return None

    def alone(self):
        """ Проверяет, что регион - единственный в стране """
        return Region.hidden_manager.filter(country=self.country).count() == 1

    def __unicode__(self):
        return self.L_title()

    def get_local_language(self):
        return self.country and self.country.language

    @property
    def name(self):
        return self.__unicode__()

    def short_title(self):
        return re.sub(u"область", u"обл.", re.sub(u"Республика", u"Респ.", self.title))

    def save(self, **kwargs):
        self.time_zone = self.time_zone or None
        super(Region, self).save(**kwargs)

    class Meta(object):
        verbose_name = _(u'область')
        verbose_name_plural = _(u'области')
        app_label = 'www'
        db_table = 'www_region'
