# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from contextlib import contextmanager

from django.db import models
from django.db.models import Q
from django.utils.translation import ugettext_lazy as _

from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement
from travel.rasp.library.python.common23.models.core.geo.station import Station


class ReplaceException(models.Model):
    u"""Исключения из колдунств [RASP-5359]"""
    city_from = models.ForeignKey(Settlement, verbose_name=_(u"город отправления"), related_name="city_from")
    city_to = models.ForeignKey(Settlement, verbose_name=_(u"город прибытия"), blank=True, null=True,
                                related_name="city_to")
    station_to = models.ForeignKey(Station, verbose_name=_(u"станция прибытия"), blank=True, null=True,
                                   related_name="station_to")

    class Meta(object):
        verbose_name = _(u'исключение колдунства')
        verbose_name_plural = _(u'исключения колдунства')
        app_label = 'www'
        db_table = 'www_replaceexception'

    _exceptions = None

    @classmethod
    def precache(cls):
        cls._exceptions = {
            (o.city_from, point_to)
            for o in cls.objects.all()
            for point_to in [_f for _f in (o.city_to, o.station_to) if _f]
        }

    @classmethod
    @contextmanager
    def using_precache(cls):
        if cls._exceptions is None:
            cls.precache()
            try:
                yield
            finally:
                cls._exceptions = None
        else:
            yield

    @classmethod
    def check_points(cls, point_from, point_to):
        if cls._exceptions is not None:
            return (point_from, point_to) in cls._exceptions
        return bool(
            cls.objects.filter(
                Q(city_from=point_from.pk) & (Q(city_to=point_to.pk) | Q(station_to=point_to.pk))
            )
        )
