# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from django.conf import settings
from django.contrib.contenttypes.fields import GenericRelation
from django.db import models
from django.utils import translation
from django.utils.translation import ugettext_lazy as _

from travel.rasp.library.python.common23.models.core.geo.city_majority import CityMajority
from travel.rasp.library.python.common23.models.core.geo.const import (
    TR_SUGGEST_FULL_TITLE_OVERRIDES, NOTHERN_CYPRUS_CITIES
)
from travel.rasp.library.python.common23.models.core.geo.country import Country
from travel.rasp.library.python.common23.models.core.geo.point import Point
from travel.rasp.library.python.common23.models.core.geo.point_slugs import (
    PointSlugField, make_settlement_slug, POINT_SLUG_LENGTH
)
from travel.rasp.library.python.common23.models.core.geo.region import Region
from travel.rasp.library.python.common23.models.core.geo.timezone_mixin import TimeZoneMixin
from travel.rasp.library.python.common23.models.core.geo.translocal_mixin import TransLocalMixin
from travel.rasp.library.python.common23.models.fields import CodeCharField
from travel.rasp.library.python.common23.models.precache.manager import PrecachingManager
from travel.rasp.library.python.common23.models.texts.geobase_ltitle import Geobase_L_title
from travel.rasp.library.python.common23.models.texts.i18n import L_field
from travel.rasp.library.python.common23.models.utils import HiddenManagerWrapper

from travel.rasp.library.python.common23.utils_db.geobase import geobase
from travel.rasp.library.python.common23.xgettext.i18n import gettext


log = logging.getLogger(__name__)


class SettlementProperty(models.Model):
    title = models.CharField(verbose_name=_(u'наименование'), max_length=100)
    description = models.TextField(verbose_name=_(u'описание'),
                                   null=True, blank=True, default=None)

    def __unicode__(self):
        return self.title

    class Meta(object):
        verbose_name = _(u'свойство города')
        verbose_name_plural = _(u'свойства городов')
        ordering = ('title',)
        app_label = 'www'
        db_table = 'www_settlementproperty'


class Settlement(models.Model, TimeZoneMixin, TransLocalMixin, Point):
    """ Город """

    # Id Москвы в нашей базе.
    # Используется как город по умолчанию во многих случаях.
    MOSCOW_ID = 213
    SPB_ID = 2
    KIEV_ID = 143
    KRASNODAR_ID = 35

    CITIES_IDS_FOR_YMAPS = {
        # Россия

        MOSCOW_ID,  # Москва (Московская область)
        SPB_ID,  # Санкт-Петербург (Ленинградская область)
        1107,  # Анапа (Краснодарский край)
        11080,  # Арзамас (Нижегородская область)
        20,  # Архангельск (Архангельская область)
        22174,  # Великий Новгород (Новгородская область)
        192,  # Владимир (Владимирская область)
        10990,  # Геленджик (Краснодарский край)
        # 25066, # Дзержинск (Житомирская область)
        972,  # Дзержинск (Нижегородская область)
        # 20692, # Дзержинск (Красноярский край)
        # 23230, # Дзержинск (Донецкая область)
        # 23261, # Дзержинск (Минская область)
        54,  # Екатеринбург (Свердловская область)
        43,  # Казань (Республика Татарстан)
        46,  # Киров (Кировская область)
        # 20200, # Киров (Калужская область)
        7,  # Кострома (Костромская область)
        KRASNODAR_ID,  # Краснодар (Краснодарский край)
        236,  # Набережные Челны (Республика Татарстан)
        23243,  # Нижний Новгород (Нижегородская область)
        65,  # Новосибирск (Новосибирская область)
        10,  # Орел (Орловская область)
        39,  # Ростов-на-Дону (Ростовская область)
        11,  # Рязань (Рязанская область)
        51,  # Самара (Самарская область)
        42,  # Саранск (Республика Мордовия)
        240,  # Тольятти (Самарская область)
        15,  # Тула (Тульская область)
        172,  # Уфа (Республика Башкортостан)
        45,  # Чебоксары (Республика Чувашия)
        56,  # Челябинск (Челябинская область)
        16,  # Ярославль (Ярославская область)

        # Украина

        KIEV_ID,  # Киев (Киев и Киевская область)
        11471,  # Алушта (Крым и г. Севастополь)
        963,  # Винница (Винницкая область)
        141,  # Днепропетровск (Днепропетровская область)
        # 11038, # Донецк (Ростовская область)
        142,  # Донецк (Донецкая область)
        11463,  # Евпатория (Крым и г. Севастополь)
        10343,  # Житомир (Житомирская область)
        960,  # Запорожье (Запорожская область)
        10345,  # Ивано-Франковск (Ивано-Франковская область)
        11464,  # Керчь (Крым и г. Севастополь)
        20221,  # Кировоград (Кировоградская область)
        23024,  # Коктебель (Крым и г. Севастополь)
        222,  # Луганск (Луганская область)
        20222,  # Луцк (Волынская область)
        144,  # Львов (Львовская область)
        10366,  # Мариуполь (Донецкая область)
        # 25811,  # Николаев (Хмельницкая область)
        148,  # Николаев (Николаевская область)
        # 22851, # Николаев (Ростовская область)
        # 23565, # Николаев (Львовская область)
        # 25721, # Новый Свет (Донецкая область)
        26094,  # Новый Свет (Крым и г. Севастополь)
        145,  # Одесса (Одесская область)
        959,  # Севастополь (Крым и г. Севастополь)
        146,  # Симферополь (Крым и г. Севастополь)
        11472,  # Судак (Крым и г. Севастополь)
        965,  # Сумы (Сумская область)
        10358,  # Ужгород (Закарпатская область)
        11469,  # Феодосия (Крым и г. Севастополь)
        147,  # Харьков (Харьковская область)
        962,  # Херсон (Херсонская область)
        # 25733, # Ялта (Донецкая область)
        11470,  # Ялта (Крым и г. Севастополь)
    }

    majority = models.ForeignKey(CityMajority, verbose_name=_(u'важность города'),
                                 null=True, blank=True, default=None)

    suggest_order = models.IntegerField(_(u'порядок в саджестах'),
                                        help_text=_(u'сначала выводятся города с меньшим значением'),
                                        null=False, default=100)

    country = models.ForeignKey(Country, verbose_name=_(u'страна'),
                                null=True, blank=True, default=None)
    region = models.ForeignKey(Region, verbose_name=_(u'область'),
                               null=True, blank=True, default=None)
    district = models.ForeignKey('District', null=True, blank=True, default=None,
                                 verbose_name=_(u"район"), related_name=u"settlements")

    L_title = Geobase_L_title(
        _(u'наименование'), add_local_field=True, local_field_critical=True, db_index=True,
        extra={
            'ru': [
                ('preposition_v_vo_na', L_field.char_field(_(u'предлог (в, во, на)'), max_length=2)),
                ('genitive', L_field.char_field(_(u'наименование (род. падеж)'))),
                ('accusative', L_field.char_field(_(u'наименование (вин. падеж)'))),
                ('locative', L_field.char_field(_(u'наименование (местн. падеж)'))),
            ]
        })

    L_abbr_title = L_field(_(u'сокр. наименование'), add_local_field=True, max_length=4,
                           help_text=_(u'максимум 4 символа'))

    slug = PointSlugField(verbose_name=_(u'название для ссылок'), max_length=POINT_SLUG_LENGTH,
                          null=True, unique=True, blank=True)

    time_zone = models.CharField(verbose_name=_(u'временная зона'),
                                 help_text=_(u'временная зона в международном формате'),
                                 null=False, blank=False, max_length=30)
    phone_info = models.CharField(verbose_name=_(u'тел. справочной'), max_length=20,
                                  null=True, blank=True, default=None)
    phone_info_short = models.CharField(verbose_name=_(u'короткий тел. справочной'),
                                        max_length=20,
                                        null=True, blank=True, default=None)
    _geo_id = models.IntegerField(verbose_name=_(u'geo ID'), null=True,
                                  blank=True, default=None, unique=True)
    _kladr_id = models.CharField(verbose_name=_(u'kladr ID'), max_length=50,
                                 null=True, blank=True, default=None,
                                 editable=False)
    sirena_id = CodeCharField(verbose_name=_(u'код в "Sirena"'), max_length=50,
                              null=True, blank=True, default=None, unique=True)
    iata = CodeCharField(verbose_name=_(u'код IATA'), max_length=20,
                         null=True, blank=True, default=None, unique=True)
    koatuu = CodeCharField(verbose_name=_(u'код КОАТУУ'), max_length=40,
                           null=True, blank=True, default=None, unique=True,
                           help_text=_(u"держа́вний класифіка́тор об'є́ктів"
                                       u" адміністрати́вно-територіа́льного у́строю Украї́ни"))
    properties = models.ManyToManyField(SettlementProperty,
                                        verbose_name=_(u'свойства'), blank=True, default=None)
    big_city = models.BooleanField(verbose_name=_(u'город на уровне столицы'),
                                   default=False)
    hidden = models.BooleanField(verbose_name=_(u'не показывать нигде!'),
                                 default=False)
    service_comment = models.TextField(verbose_name=_(u'служебный комментарий'),
                                       help_text=_(u'не показывается в сервисе'),
                                       null=True, blank=True, default=None)
    has_tablo = models.BooleanField(verbose_name=_(u'город имеет табло'),
                                    default=True, blank=False, null=False)
    agent_geo_id = models.IntegerField(verbose_name=_(u'agent Geo ID'), null=True,
                                       blank=True, default=None)
    suburban_zone = models.ForeignKey("SuburbanZone", blank=True, null=True,
                                      verbose_name=_(u"пригородная зона"),
                                      help_text=_(u"пригородная зона, в которую входит город"),
                                      related_name="settlement_set")
    has_many_airports = models.BooleanField(verbose_name=_(u'много аэропортов'),
                                            default=False, null=False)
    has_urban_transport = models.BooleanField(verbose_name=_(u'есть городской транспорт'),
                                              default=False, null=False)
    human_url = CodeCharField(verbose_name=_(u'код города для ЧПУ'), max_length=50,
                              null=True, blank=True, default=None, unique=True)
    threads_amount = models.PositiveIntegerField(verbose_name=_(u'общее количество рейсов в табло'),
                                                 default=0, editable=False)
    synonyms = GenericRelation('PointSynonym')
    type_choices = models.CharField(verbose_name=_(u'типы ТС'), max_length=50, default='', null=False, blank=True)
    longitude = models.FloatField(verbose_name=_(u"долгота"), default=None, blank=True, null=True,
                                  db_index=True)
    latitude = models.FloatField(verbose_name=_(u"широта"), default=None, blank=True, null=True,
                                 db_index=True)
    _disputed_territory = models.BooleanField(verbose_name=_(u"спорная территория"), default=False,
                                              db_column='disputed_territory')
    modified_at = models.DateTimeField(_(u'Дата-время изменения'), auto_now=True, null=False, blank=True)

    use_in_suburban_app_suggests = models.CharField(default='no', max_length=20,
                                                    verbose_name=u'использовать в саджестах приложения электрички',
                                                    choices=[('all_railway_stations', u'c припиской "все вокзалы"'),
                                                             ('yes', u'да'),
                                                             ('no', u'нет')])

    objects = PrecachingManager(keys=['_geo_id', 'pk'], iexact_keys=['iata'])

    hidden_manager = HiddenManagerWrapper('objects')

    @property
    def type_choices_set(self):
        return self.type_choices and set(self.type_choices.split(',')) or set()

    @property
    def title_with_prefix(self):
        if self.majority_id != 5:  # Поселок или деревня
            return u"г. " + self.title

        return self.title

    def L_title_with_prefix(self, lang=None):
        if not lang:
            lang = translation.get_language()

        if self.majority_id != 5:  # Поселок или деревня
            return gettext(u'г. %s', lang=lang) % self.L_title(lang=lang)

        return self.L_title(lang=lang)

    def L_title_with_type(self, lang=None):
        return self.L_title_with_prefix(lang)

    def get_geobase_region(self):
        if not self._geo_id or not geobase:
            return None

        try:
            return geobase.region_by_id(self._geo_id)
        except Exception:
            log.exception(u'Не удалось получить ename города %s из геобазы', self._geo_id)
            return None

    @classmethod
    def get_default_city(cls):
        """Дефолт-сити"""
        return cls.hidden_manager.get(id=Settlement.MOSCOW_ID)

    @classmethod
    def get_default_for_root_domain(cls, root_domain):
        if root_domain in settings.DEFAULT_CITIES:
            return cls.objects.get(id=settings.DEFAULT_CITIES[root_domain])

        try:
            country = Country.objects.get(domain_zone=root_domain)

            return country.get_capital()
        except Country.DoesNotExist:
            pass

        return cls.get_default_city()

    def save(self, **kwargs):
        self.time_zone = self.time_zone or None
        if not self.country_id:
            if self.region:
                self.country = self.region.country
        if not self.slug:
            self.slug = make_settlement_slug(self)
        else:
            self.slug = self.slug.lower()

        super(Settlement, self).save(**kwargs)
        # если это insert, то иногда уникальный slug можно получить только при известном id
        if not self.slug:
            self.slug = make_settlement_slug(self)
            if self.slug:
                super(Settlement, self).save()

    def __unicode__(self):
        reg = ""
        cntr = ""

        # Чтобы это бралось из кэша
        try:
            region = self.region_id and Region.objects.get(pk=self.region_id) or None
        except Region.DoesNotExist:
            region = None

        try:
            country = self.country_id and Country.objects.get(pk=self.country_id) or None
        except Country.DoesNotExist:
            country = None

        if region:
            reg = u" (" + region.L_title() + u")"
        elif country:
            cntr = u" (" + country.L_title() + u")"
        return self.L_title() + (reg or cntr)

    def get_local_language(self):
        return self.country and self.country.language

    @property
    def name(self):
        return self.title

    def get_popular_title(self, **kwargs):
        return self.L_title()

    def get_short_title(self):
        return self.L_title()

    def L_omonim_title(self, lang=None, show_district=True, national_version=None):
        u"""Название города для саджестов"""
        result = {
            'title': self.L_title_with_prefix(lang),
            'add': ''
        }

        add = []
        if show_district and self.district_id:
            add.append(self.district.title)

        if self.region_id:
            add.append(self.region.L_title(lang=lang))

        country = self.translocal_country(national_version)
        if country:
            add.append(country.L_title(lang=lang))

        if add:
            result['add'] = u", ".join(add)

        return result

    def L_omonim_title_bem(self, request, direction, **extra_params):
        u"""Название города со ссылкой для переключения в омонимах"""
        result = {
            'title': self.L_title_with_prefix(),
        }

        add = []

        if self.district_id:
            add.append(self.district.title)

        if self.region_id:
            add.append(self.region.L_title())

        if self.country_id and not self.disputed_territory:
            add.append(self.country.L_title())

        if add:
            result['add'] = add

        return result

    def L_full_geography(self, show_district=False, national_version=None, lang=None):
        country = self.translocal_country(national_version)

        # Для Москвы, Санкт-Петербурга и Киева регион не указывается,
        # так как регион называется "Москва и Московская область"
        cities_as_regions = [self.MOSCOW_ID, self.SPB_ID, self.KIEV_ID]
        is_region = self.id in cities_as_regions

        breadcrumbs = [
            show_district and self.district and not is_region and self.district.L_title_with_postfix(lang=lang),
            self.region and not is_region and self.region.L_title(lang=lang),
            country and country.L_title(lang=lang)
        ]

        return u', '.join(b for b in breadcrumbs if b)

    def L_title_with_full_geography(self, national_version=None, lang=None):
        """
        Получает строку с минимально допустимым количеством коллизий
        для поиска населенного пункта в сторонних сервисах: blablacar.ru, maps.yandex.ru
        Примеры:
        Москва, Россия
        Первоуральск, Свердловская область, Россия
        Первомайский, Мамонтовский р-н, Алтайский край, Россия
        """
        parts = [self.L_title(lang=lang), self.L_full_geography(True, national_version, lang)]
        return u', '.join(part for part in parts if part)

    def L_suggest_full_title_tr(self, lang=None):
        """Full title for turkish suggests"""

        override = TR_SUGGEST_FULL_TITLE_OVERRIDES.get(self.point_key)

        if override:
            return override

        parts = [
            self.L_title_with_type(lang),
        ]

        if self.region_id and self.majority_id > CityMajority.POPULATION_MILLION_ID:
            parts.append(self.region.L_title(lang=lang))

        country = self.translocal_country('tr')

        if country:
            parts.append(country.L_title(lang=lang))

        return parts

    @property
    def disputed_territory(self):
        """RASP-15442"""
        if self._disputed_territory:
            return True

        if self.region_id and self.region.disputed_territory:
            return True

        return self.district_id and self.district.disputed_territory

    class Meta(object):
        verbose_name = _(u'город')
        verbose_name_plural = _(u'города')
        app_label = 'www'
        db_table = 'www_settlement'

    is_city = True

    def city_size(self):
        return self.majority_id and (
            (self.majority_id in [1, 2] and 'big') or (self.majority_id in [5] and 'small') or ''
        ) or ''

    def is_foreign(self, national_version=None):
        if national_version == 'tr' and self.id in NOTHERN_CYPRUS_CITIES:
            return False
        if self.country_id:
            return self.country_id not in settings.OUR_COUNTRIES
        if self.region and self.region.country_id:
            return self.region.country_id not in settings.OUR_COUNTRIES
        return True

    def has_actual_tablo(self):
        """Для городов актуальное табло не строится"""
        return False

    def get_agent_geo_id(self):
        """Определяет agent_geo_id у города"""
        agent_geo_id = self.agent_geo_id

        if agent_geo_id is None:
            agent_geo_id = self.region and self.region.agent_geo_id

        return agent_geo_id

    def get_externaldirections(self):
        from travel.rasp.library.python.common23.models.core.directions.external_direction import ExternalDirection
        return ExternalDirection.objects.filter(externaldirectionmarker__station__settlement=self).distinct()

    def get_directions(self):
        """
        main_directions - если есть пригородная зона, то направления, проходящие через станции города
        directions - остальные направления, либо все, если город не привязан к пригородной зоне
        """

        if self.suburban_zone:
            directions = self.suburban_zone.externaldirection_set.all()
            main_directions = list(directions.filter(externaldirectionmarker__station__settlement=self).distinct())
            # Тут только не проходящие через станции города
            directions = [d for d in directions if d not in main_directions]

            return main_directions, directions

        return [], []

    @property
    def has_ymap(self):
        return self.id in Settlement.CITIES_IDS_FOR_YMAPS

    @property
    def tz_name_abbr(self):
        tz_name = self.title_abbr or self.title_ru_locative or self.title

        return u'время в %s' % tz_name
